<?php

namespace App\Console\Commands;

use App\Http\Controllers\FcmController;
use Illuminate\Console\Command;
use App\Models\Installment;
use App\Models\User;
use Carbon\Carbon;

class AutoDeductInstallments extends Command
{
    // Define the name and signature of the command.
    protected $signature = 'installments:auto-deduct';

    // Command description
    protected $description = 'Automatically deduct installments on due dates';

    // Handle the logic for auto-deducting the installments
    public function handle(): void
    {
        // Get the current date
        $currentDate = Carbon::now();

        // Fetch all installments that are due and unpaid
        $dueInstallments = Installment::where('status', 'unpaid')
            ->where('due_date', '<=', $currentDate)  // Check if due date has passed or is today
            ->get();

        foreach ($dueInstallments as $installment) {
            // Assuming Installment is related to a loan, which is related to a user
            $user = $installment->loan->user;

            // Check if the user has enough balance to pay the installment
            if ($user->balance >= $installment->per_installment) {
                // Deduct the installment amount from the user's balance
                $user->balance -= $installment->per_installment;
                $user->save();

                // Update the installment status to "paid"
                $installment->status = 'paid';
                $installment->update();

                // Increment the count of paid installments for the loan
                $installment->loan->paid_installment += 1;
                $installment->loan->update();

                // Log the payment info
                $this->info('Installment ' . $installment->installment_id . ' paid for user: ' . $user->name);
            } else {
                // If the user doesn't have enough balance, log a warning
//                $overDue = OverDue::where('bill_id', $installment->installment_id)->first();
//                if (!$overDue) {
//                    OverDue::create([
//                        'user_id' => $user->id,
//                        'bill_id' => $installment->installment_id,
//                        'amount' => $installment->per_installment,
//                        'due_date' => $installment->due_date,
//                    ]);
//                }

                $notification = new FcmController();
                $notification->sendFcmNotification($user->id, 'Installment Overdue', $installment->installment_id . ' paid for user: ' . $user->name);
                $this->warn('Insufficient balance for user: ' . $user->name . ' for installment: ' . $installment->installment_id);
            }

            // After processing all due installments, check if the loan is ready to be closed
            $unpaidInstallments = Installment::where('loan_application_id', $installment->loan->id)
                ->where('status', 'unpaid')
                ->count();

            if ($unpaidInstallments === 0) {
                // If no unpaid installments remain, mark the loan as closed
                $installment->loan->status = 'closed';
                $installment->loan->update();

                $this->info("Loan application {$installment->loan->id} is now closed.");
            }
        }

        // Inform that the auto-deduction process is complete
        $this->info('Auto deduction process completed.');
    }

}
