<?php

namespace App\Console\Commands;

use App\Models\ApiRecharge;
use App\Models\Recharge;
use App\Models\User;
use GuzzleHttp\Client;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class CheckRechargeStatus extends Command
{
    protected $signature = 'recharge:check-status';
    protected $description = 'Check recharge status from the API and update accordingly';

    public function handle(): void
    {
        $client = new Client();
        $api_info = ApiRecharge::first();

        if (!$api_info) {
            Log::warning('No API info found in ApiRecharge table.');
            return;
        }

        $pendingRecharges = Recharge::where('status', 'pending')->get();

        foreach ($pendingRecharges as $recharge) {
            try {
                $response = $client->post($api_info->api_url . "/api/recharge/verify", [
                    'headers' => [
                        'Content-Type' => 'application/json',
                        'CD-API-KEY' => $api_info->api_key,
                    ],
                    'json' => [
                        'ref_id' => $recharge->trx_id,
                    ]
                ]);

                $result = json_decode($response->getBody(), true);
                Log::info("API Response for trx_id {$recharge->trx_id}", $result);

                if (!isset($result['data']['status'])) {
                    Log::warning("No status found in API response for trx_id: {$recharge->trx_id}");
                    continue;
                }

                $apiRechargeStatus = strtolower($result['data']['status']);
                $user = User::find($recharge->user_id);

                if (!$user) {
                    Log::warning("User not found for trx_id: {$recharge->trx_id}");
                    continue;
                }

                if ($apiRechargeStatus === 'completed') {
                    $recharge->status = 'accepted';
                    $recharge->trx_id = $result['data']['trx_id'] ?? $recharge->trx_id;
                    $recharge->save();
                    Log::info("Recharge accepted for trx_id: {$recharge->trx_id}");
                } elseif ($apiRechargeStatus === 'cancelled') {
                    $user->balance += $recharge->amount;

                    if ($recharge->commission > 0) {
                        $user->balance -= $recharge->commission;
                        Log::info("Commission refunded: {$recharge->commission} for trx_id: {$recharge->trx_id}");
                    }

                    $user->save();

                    $recharge->status = 'rejected';
                    $recharge->save();

                    Log::info("Recharge rejected and balance refunded for trx_id: {$recharge->trx_id}");
                } else {
                    Log::info("Recharge still pending or unknown status for trx_id: {$recharge->trx_id}");
                }

            } catch (\Exception $e) {
                Log::error("Recharge status check failed for trx_id: {$recharge->trx_id}. Error: " . $e->getMessage());
            }
        }

        Log::info('Recharge status check completed.');
    }
}