<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Controllers\FcmController;
use App\Models\Bank;
use App\Models\BankTransfer;
use App\Models\Settings;
use App\Models\User;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Illuminate\View\View;

class BankingController extends Controller
{
    //================>> Bank <<===============
    public function index(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $bank_data = Bank::all()->sortByDesc('created_at');
        return view('banking.index', compact('dashboard', 'settings', 'bank_data'));
    }

    public function addBank(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        return view('banking.add', compact('dashboard', 'settings'));
    }

    public function storeBank(Request $request): RedirectResponse
    {
        $validator = Validator::make(request()->all(), [
            'name' => 'required|unique:banks,name',
            'logo' => 'required|image|mimes:jpeg,png,jpg,|max:2048',
        ], [
            'name.required' => 'Please enter Bank name',
            'name.unique' => 'Bank name already exists',
            'logo.required' => 'Please enter Bank logo',
            'logo.image' => 'Please upload logo',
            'logo.mimes' => 'Please upload JPG, PNG, JPEG',
            'logo.max' => 'Maximum upload file size 2MB',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $store = new Bank();
        $store->name = $request->name;
        $store->slug = Str::slug($request->name);

        if ($request->hasFile('logo')) {
            $file = $request->file('logo');
            $fileName = time() . '.' . $file->getClientOriginalExtension();
            $path = 'upload/banks/' . $fileName;
            $file->move(public_path('upload/banks/'), $fileName);
            $store->logo = $path; // Store the file path in the database
        }

        $status = $request->has('status') == 'on' ? true : false;
        $store->status = $status;
        $store->save();

        $notification = [
            'message' => $request->name . ' added successfully',
            'alert-type' => 'success',
        ];

        return back()->with($notification);
    }

    public function editBank($id): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $bank = Bank::findOrFail($id);
        return view('banking.edit', compact('dashboard', 'settings', 'bank'));
    }

    public function updateBank(Request $request, $id): RedirectResponse
    {
        $validator = Validator::make(request()->all(), [
            'name' => ['nullable', Rule::unique('banks')->ignore($id)],
            'slug' => ['nullable', Rule::unique('banks')->ignore($id)],
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,|max:2048',
        ], [
            'name.required' => 'Please enter mobile bank name',
            'name.unique' => 'Mobile bank name already exists',
            'logo.required' => 'Please enter mobile bank logo',
            'logo.image' => 'Please upload logo',
            'logo.mimes' => 'Please upload JPG, PNG, JPEG',
            'logo.max' => 'Maximum upload file size 2MB',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $update = Bank::findOrFail($id);
        $update->name = $request->name;
        $update->slug = Str::slug($request->name);

        if ($request->hasFile('logo')) {

            if ($update->logo && File::exists(public_path($update->logo))) {
                File::delete(public_path($update->logo));
            }

            $file = $request->file('logo');
            $fileName = time() . '.' . $file->getClientOriginalExtension();
            $path = 'upload/banks/' . $fileName;
            $file->move(public_path('upload/banks/'), $fileName);
            $update->logo = $path; // Store the file path in the database
        }

        $status = $request->has('status') == 'on' ? true : false;
        $update->status = $status;
        $update->update();

        $notification = [
            'message' => $request->name . ' updated successfully',
            'alert-type' => 'success',
        ];

        return back()->with($notification);
    }

    public function deleteBank($id): RedirectResponse
    {
        $deleteBank = Bank::findOrFail($id);
        if ($deleteBank->logo && File::exists(public_path($deleteBank->logo))) {
            File::delete(public_path($deleteBank->logo));
        }
        $deleteBank->delete();
        $notification = [
            'message' => $deleteBank->name . ' deleted successfully',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);
    }

    //============= Banking actions
    public function bankingPending(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $items = BankTransfer::where('status', 'pending')->latest()->get();
        return view('banking.banking_order', compact('dashboard', 'settings', 'items'));
    }

    public function bankingSuccess(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $items = BankTransfer::where('status', 'approved')->latest()->get();
        return view('banking.banking_order', compact('dashboard', 'settings', 'items'));
    }

    public function bankingRejected(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $items = BankTransfer::where('status', 'rejected')->latest()->get();
        return view('banking.banking_order', compact('dashboard', 'settings', 'items'));
    }

    public function bankingApproved(Request $request): RedirectResponse
    {
        $notificationController = new FcmController();

        $update = BankTransfer::where('id', $request->success_id)->first();
        $update->update([
            'status' => 'approved',
            'ac_number' => $request->change_number ?? $update->ac_number,
        ]);

        $notificationController->sendFcmNotification($update->user_id, $update->bank_name, 'Your ' . $update->bank_name . ' service is approved successfully');

        $notification = [
            'message' => 'This ' . $update->bank_name . ' service is approved successfully',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);

    }

    public function bankingOrderRejected(Request $request): RedirectResponse
    {
        $notificationController = new FcmController();

        $validator = Validator::make($request->all(), [
            'reason' => 'required'
        ], [
            'reason.required' => 'Write Mobile Banking Rejected Reason',
        ]);

        if ($validator->fails()) {

            $notification = [
                'message' => $validator->errors()->first(),
                'alert-type' => 'error'
            ];

            return redirect()->back()->with($notification);
        }

        $update = BankTransfer::where('id', $request->reject_id)->first();
        $update->update([
            'status' => 'rejected',
            'reason' => $request->reason
        ]);

        $refundAmount = $update->amount + $update->fee;

        $update->user->balance += $refundAmount;
        $update->user->balance -= $update->commission;
        $update->user->update();

        $notificationController->sendFcmNotification($update->user_id, $update->bank_name, 'Your ' . $update->bank_name . ' service is rejected & refunded successfully');


        $notification = [
            'message' => 'This ' . $update->bank_name . ' service is rejected successfully!',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);

    }

    public function bankingDelete($id): RedirectResponse
    {
        $delete = BankTransfer::findOrFail($id);
        $delete->delete();

        Log::info($id);

        $notification = [
            'message' => 'This service is deleted successfully!',
            'alert-type' => 'success'
        ];
        return redirect()->back()->with($notification);
    }

}
