<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Controllers\FcmController;
use App\Models\Billing;
use App\Models\BillPayment;
use App\Models\Settings;
use App\Models\User;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Illuminate\View\View;

class BillController extends Controller
{
    public function index(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $billing_data = Billing::all()->sortByDesc('created_at');
        return view('bill.index', compact('dashboard', 'settings', 'billing_data'));
    }

    public function addBill(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        return view('bill.add', compact('dashboard', 'settings'));
    }

    public function storeBill(Request $request): RedirectResponse
    {
        $validator = Validator::make(request()->all(), [
            'name' => 'required|unique:billings,name',
            'type' => 'required',
            'logo' => 'required|image|mimes:jpeg,png,jpg,|max:2048',
        ], [
            'name.required' => 'Please enter Billing name',
            'name.unique' => 'Billing name already exists',
            'type.required' => 'Please enter Billing type',
            'logo.required' => 'Please enter Billing logo',
            'logo.image' => 'Please upload logo',
            'logo.mimes' => 'Please upload JPG, PNG, JPEG',
            'logo.max' => 'Maximum upload file size 2MB',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $store = new Billing();
        $store->name = $request->name;
        $store->type = $request->type;
        $store->slug = Str::slug($request->name);

        if ($request->hasFile('logo')) {
            $file = $request->file('logo');
            $fileName = time() . '.' . $file->getClientOriginalExtension();
            $path = 'upload/billings/' . $fileName;
            $file->move(public_path('upload/billings/'), $fileName);
            $store->logo = $path; // Store the file path in the database
        }

        $status = $request->has('status') == 'on' ? true : false;
        $store->status = $status;
        $store->save();

        $notification = [
            'message' => $request->name . ' added successfully',
            'alert-type' => 'success',
        ];

        return back()->with($notification);
    }

    public function editBill($id): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $bill = Billing::findOrFail($id);
        return view('bill.edit', compact('dashboard', 'settings', 'bill'));
    }

    public function updateBill(Request $request, $id): RedirectResponse
    {
        $validator = Validator::make(request()->all(), [
            'name' => ['required', Rule::unique('billings')->ignore($id)],
            'type' => 'required',
            'slug' => ['nullable', Rule::unique('billings')->ignore($id)],
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,|max:2048',
        ], [
            'name.required' => 'Please enter mobile bank name',
            'name.unique' => 'Mobile bank name already exists',
            'logo.required' => 'Please enter mobile bank logo',
            'logo.image' => 'Please upload logo',
            'logo.mimes' => 'Please upload JPG, PNG, JPEG',
            'logo.max' => 'Maximum upload file size 2MB',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $update = Billing::findOrFail($id);
        $update->name = $request->name;
        $update->type = $request->type;
        $update->slug = Str::slug($request->name);

        if ($request->hasFile('logo')) {

            if ($update->logo && File::exists(public_path($update->logo))) {
                File::delete(public_path($update->logo));
            }

            $file = $request->file('logo');
            $fileName = time() . '.' . $file->getClientOriginalExtension();
            $path = 'upload/billings/' . $fileName;
            $file->move(public_path('upload/billings/'), $fileName);
            $update->logo = $path; // Store the file path in the database
        }

        $status = $request->has('status') == 'on' ? true : false;
        $update->status = $status;
        $update->update();

        $notification = [
            'message' => $request->name . ' updated successfully',
            'alert-type' => 'success',
        ];

        return back()->with($notification);
    }

    public function deleteBill($id): RedirectResponse
    {
        $deleteBill = Billing::findOrFail($id);
        if ($deleteBill->logo && File::exists(public_path($deleteBill->logo))) {
            File::delete(public_path($deleteBill->logo));
        }
        $deleteBill->delete();
        $notification = [
            'message' => $deleteBill->name . ' deleted successfully',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);
    }

    //============= Bill actions
    public function billPending(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $items = BillPayment::where('status', 'pending')->latest()->get();
        return view('bill.bill_order', compact('dashboard', 'settings', 'items'));
    }

    public function billSuccess(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $items = BillPayment::where('status', 'approved')->latest()->get();
        return view('bill.bill_order', compact('dashboard', 'settings', 'items'));
    }

    public function billRejected(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $items = BillPayment::where('status', 'rejected')->latest()->get();
        return view('bill.bill_order', compact('dashboard', 'settings', 'items'));
    }

    public function billApproved(Request $request): RedirectResponse
    {
        $notificationController = new FcmController();

        $update = BillPayment::where('id', $request->success_id)->first();
        $update->update([
            'status' => 'approved'
        ]);

        $notificationController->sendFcmNotification($update->user_id, $update->billing_name, 'Your ' . $update->billing_name . ' service is approved');

        $notification = [
            'message' => 'This ' . $update->billing_name . ' service is approved successfully',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);

    }

    public function billOrderRejected(Request $request): RedirectResponse
    {
        $notificationController = new FcmController();

        $validator = Validator::make($request->all(), [
            'reason' => 'required'
        ], [
            'reason.required' => 'Write Mobile Banking Rejected Reason',
        ]);

        if ($validator->fails()) {

            $notification = [
                'message' => $validator->errors()->first(),
                'alert-type' => 'error'
            ];

            return redirect()->back()->with($notification);
        }

        $update = BillPayment::where('id', $request->reject_id)->first();
        $update->update([
            'status' => 'rejected',
            'reason' => $request->reason
        ]);

        $refundAmount = $update->amount + $update->fee;

        $update->user->balance += $refundAmount;
        $update->user->balance -= $update->commission;
        $update->user->update();

        $notificationController->sendFcmNotification($update->user_id, $update->billing_name, 'Your ' . $update->billing_name . ' service is rejected & refunded successfully');

        $notification = [
            'message' => 'This ' . $update->billing_name . ' service is rejected successfully!',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);

    }

    public function billDelete($id): RedirectResponse
    {
        $delete = BillPayment::findOrFail($id);
        $delete->delete();

        $notification = [
            'message' => 'This service is deleted successfully!',
            'alert-type' => 'success'
        ];
        return redirect()->back()->with($notification);
    }

}
