<?php

namespace App\Http\Controllers\Admin;

use App\Helpers\GreetingHelper;
use App\Http\Controllers\Controller;
use App\Models\AddMoneyOrder;
use App\Models\BankTransfer;
use App\Models\BillPayment;
use App\Models\Commission;
use App\Models\Company;
use App\Models\Config;
use App\Models\DriveOrder;
use App\Models\Extra;
use App\Models\GlobalRechargeOrder;
use App\Models\LoanApplications;
use App\Models\MBankingOrder;
use App\Models\NoticeMarquee;
use App\Models\Offer;
use App\Models\Permission;
use App\Models\PromotionBanner;
use App\Models\Recharge;
use App\Models\RemittanceOrder;
use App\Models\SendmoneyOrder;
use App\Models\Settings;
use App\Models\SmsSetting;
use App\Models\Support;
use App\Models\User;
use App\Models\Videos;
use App\Models\VideoTutorial;
use Google\Service\Forms\Video;
use Google\Service\Spanner\Transaction;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\View\View;
use PHPUnit\Framework\TestStatus\Notice;

class DashboardController extends Controller
{
    public function index(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $greeting = GreetingHelper::greetUser($dashboard->name);
        $settings = Settings::findOrFail(1);
        $rechargePending = Recharge::where("status", 'pending')->count();
        $addmoneyPending = AddMoneyOrder::where("status", 'pending')->count();
        $bankTransferPending = BankTransfer::where("status", 'pending')->count();
        $billPaymentsPending = BillPayment::where("status", 'pending')->count();
        $mBankingOrderPending = MBankingOrder::where("status", 'pending')->count();
        $remittanceOrderPending = RemittanceOrder::where("status", 'pending')->count();

        return view('index', compact('dashboard', 'settings', 'greeting', 'rechargePending',
            'addmoneyPending', 'bankTransferPending', 'billPaymentsPending', 'mBankingOrderPending', 'remittanceOrderPending'));
    }


    //===========>> Profile <<=========//
    public function profileIndex(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        return view('profile.index', compact('dashboard', 'settings'));
    }

    public function updateProfile(Request $request): RedirectResponse
    {
        $validator = Validator::make($request->all(), [
            'username' => 'required|unique:users,username,' . auth()->id(),
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users,email,' . auth()->id(),
            'phone' => 'required|max:255|unique:users,phone,' . auth()->id(),
            'location' => 'required|string|max:255',
            'device' => 'required|string|max:255',
            'profile' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ], [
            'username.required' => 'Username is required',
            'username.unique' => 'Username is already taken',
            'name.required' => 'Name is required',
            'name.max' => 'Name cannot be longer than 255 characters',
            'email.required' => 'Email is required',
            'email.unique' => 'Email is already taken',
            'email.string' => 'Email must be a string',
            'email.max' => 'Email cannot be longer than 255 characters',
            'email.email' => 'Email must be a valid email',
            'phone.required' => 'Phone is required',
            'phone.max' => 'Phone cannot be longer than 255 characters',
            'phone.unique' => 'Phone is already taken',
            'location.required' => 'Location is required',
            'location.max' => 'Location cannot be longer than 255 characters',
            'device.required' => 'Device is required',
            'device.max' => 'Device cannot be longer than 255 characters',
            'profile.image' => 'Image must be a image',
            'profile.mimes' => 'Image must be a file of type: jpeg, png, jpg',
            'profile.max' => 'Image must be less than 2MB',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $update = User::findOrFail(auth()->id());
        $update->username = $request->username;
        $update->name = $request->name;
        $update->email = $request->email;
        $update->phone = $request->phone;
        $update->location = $request->location;
        $update->device = $request->device;

        if ($request->hasFile('profile')) {
            if ($update->profile && File::exists(public_path($update->profile))) {
                File::delete(public_path($update->profile));
            }
            $update->profile = $this->processImage($request->file('profile'), 'profile');
        }

        $update->update();

        $notification = [
            'message' => 'Profile updated successfully',
            'alert-type' => 'success',
        ];

        return redirect()->back()->with($notification);

    }

    public function updatePassword(Request $request): RedirectResponse
    {
        $validator = Validator::make($request->all(), [
            'old_password' => 'required',
            'new_password' => 'required|string|min:8',
            'confirm_password' => 'required|string|min:8|same:new_password',
        ], [
            'old_password.required' => 'Old Password is required',
            'new_password.required' => 'New Password is required',
            'new_password.min' => 'New Password must be at least 8 characters',
            'confirm_password.required' => 'Confirm Password is required',
            'confirm_password.min' => 'Confirm Password must be at least 8 characters',
            'confirm_password.same' => 'Confirm Password does not match',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator);
        }

        if (!Hash::check($request->old_password, auth::user()->password)) {
            $notification = [
                'message' => 'Old password does not match',
                'alert-type' => 'warning',
            ];
            return redirect()->back()->withErrors(['old_password' => 'Old password is incorrect'])->with($notification);
        }

        $updatePass = User::findOrFail(auth()->id());
        $updatePass->password = Hash::make($request->confirm_password);
        $updatePass->update();

        $notification = [
            'message' => 'Password updated successfully',
            'alert-type' => 'success',
        ];
        return redirect()->back()->with($notification);
    }

    //todo: Start support Index

    public function supportIndex(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $support = Support::findOrFail(1);
        $extra = Extra::findOrFail(1);
        return view('support.index', compact('dashboard', 'settings', 'support', 'extra'));
    }

    public function updateSupport(Request $request): RedirectResponse
    {
        Support::findOrFail(1)->update([
            'messenger' => $request->messenger_url,
            'telegram' => $request->tg_username,
            'whatsapp' => $request->whatsapp_number,
            'youtube' => $request->youtube_url,
            'about' => $request->about,
        ]);

        $notification = [
            'message' => 'Support updated successfully',
            'alert-type' => 'success',
        ];
        return redirect()->back()->with($notification);
    }

    public function updateExtra(Request $request)
    {
        Extra::findOrFail(1)->update([
            'web_address' => $request->web_address,
            'tutorial' => $request->tutorial,
        ]);


        $notification = [
            'message' => 'Links updated successfully',
            'alert-type' => 'success',
        ];
        return redirect()->back()->with($notification);
    }

    //todo: Start Settings Index
    public function settingsIndex(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $config = Config::findOrFail(1);
        return view('settings.index', compact('dashboard', 'settings', 'config'));
    }

    public function updateBrand(Request $request): RedirectResponse
    {
        $validator = Validator::make($request->all(), [
            'brandName' => 'required',
            'terms_url' => 'required',
            'privacy_url' => 'required',
            'brandLogo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'brandFavIcon' => 'nullable',
        ], [
            'brandName.required' => 'Brand Name is required.',
            'brandLogo.image' => 'Brand Logo must be an image.',
            'brandLogo.mimes' => 'Brand Logo must be a file of type: jpeg, png, jpg.',
            'brandLogo.max' => 'Brand Logo must be less than 2MB.',
            'brandFavIcon.image' => 'Favicon must be an image.',
            'brandFavIcon.mimes' => 'Favicon must be a file of type: jpeg, png, jpg.',
            'brandFavIcon.max' => 'Favicon must be less than 2MB.',
            'brandFavIcon.dimensions' => 'Favicon size must be 80x80px.',
            'brandLogo.dimensions' => 'BrandLogo must be less than 187x32px.',
            'terms_url.required' => 'Terms & Conditions is required.',
            'privacy_url.required' => 'Privacy & Terms & Conditions are required.',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $update = Settings::findOrFail(1);
        $update->brandName = $request->brandName;
        $update->terms_url = $request->terms_url;
        $update->privacy_url = $request->privacy_url;

        if ($request->hasFile('brandLogo')) {
            if ($update->brandLogo && File::exists(public_path($update->brandLogo))) {
                File::delete(public_path($update->brandLogo));
            }
            $update->brandLogo = $this->processImage($request->file('brandLogo'), 'brandLogo');
        }
        if ($request->hasFile('brandFavIcon')) {
            if ($update->brandFavicon && File::exists(public_path($update->brandFavicon))) {
                File::delete(public_path($update->brandFavicon));
            }
            $update->brandFavicon = $this->processImage($request->file('brandFavIcon'), 'brandFavIcon');
        }
        $update->save();
        $notification = [
            'message' => 'Settings Updated Successfully',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);
    }


    private function processImage($file, $fieldName): string
    {
        $fileName = time() . '_' . $fieldName . '.' . $file->getClientOriginalExtension();
        $path = 'upload/admins/' . $fileName;
        $file->move(public_path('upload/admins/'), $fileName);
        return $path;
    }


    //todo: Start Promotion Function
    public function promotionIndex(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $banners = PromotionBanner::all();
        $notice = NoticeMarquee::findOrFail(1);
        $dNotice = \App\Models\Notice::findOrFail(1);
        return view('promotion.index', compact('dashboard', 'settings', 'banners', 'notice', 'dNotice'));
    }

    public function storePromotion(Request $request): RedirectResponse
    {

        $validator = Validator::make($request->all(), [
            'banner' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048', // Adjust the validation rules as needed
        ], [
            'banner.required' => 'Banner is required.',
            'banner.image' => 'Banner must be an image.',
            'banner.mimes' => 'Banner must be a file of type: jpeg, png, jpg.',
            'banner.max' => 'Banner must be less than 2MB.',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $store = new PromotionBanner();

        if ($request->hasFile('banner')) {
            $store->banner = $this->processBanner($request->file('banner'), 'banner');
        }
        $store->save();

        $notification = [
            'message' => 'Banner added successfully',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);

    }

    private function processBanner($file, $fieldName): string
    {
        $fileName = time() . '_' . $fieldName . '.' . $file->getClientOriginalExtension();
        $path = 'upload/banners/' . $fileName;
        $file->move(public_path('upload/banners/'), $fileName);
        return $path;
    }

    public function promotionDelete($id): RedirectResponse
    {
        $delete = PromotionBanner::findOrFail($id);
        if ($delete->banner && File::exists(public_path($delete->banner))) {
            File::delete(public_path($delete->banner));
        }
        $delete->delete();
        $notification = [
            'message' => 'Promotion banner deleted successfully',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);
    }

    public function deleteOffer($id): RedirectResponse
    {
        $delete = Offer::findOrFail($id);
        if ($delete->image && File::exists(public_path($delete->image))) {
            File::delete(public_path($delete->image));
        }
        $delete->delete();
        $notification = [
            'message' => $delete->title . ' deleted successfully',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);
    }


    public function promotionMarquee(Request $request): RedirectResponse
    {
        $validator = Validator::make($request->all(), [
            'notice' => 'required',
        ], [
            'notice.required' => 'Notice is required.',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $marquee = NoticeMarquee::findOrFail(1);
        $marquee->notice_text = $request->notice;
        $marquee->update();

        $notification = [
            'message' => 'Notice updated successfully',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);
    }

    public function dashboardNotice(Request $request): RedirectResponse
    {
        $validator = Validator::make($request->all(), [
            'notice' => 'required',
        ], [
            'notice.required' => 'Notice is required.',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $marquee = \App\Models\Notice::findOrFail(1);
        $marquee->notice = $request->notice;
        $marquee->update();

        $notification = [
            'message' => 'Notice updated successfully',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);
    }

    public function packageNotice(Request $request): RedirectResponse
    {
        $validator = Validator::make($request->all(), [
            'notice' => 'required',
        ], [
            'notice.required' => 'Notice is required.',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $marquee = \App\Models\Notice::findOrFail(1);
        $marquee->package_notice = $request->notice;
        $marquee->update();

        $notification = [
            'message' => 'Notice updated successfully',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);
    }

    public function rateNotice(Request $request): RedirectResponse
    {
        $validator = Validator::make($request->all(), [
            'notice' => 'required',
        ], [
            'notice.required' => 'Notice is required.',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $marquee = \App\Models\Notice::findOrFail(1);
        $marquee->rate_notice = $request->notice;
        $marquee->update();

        $notification = [
            'message' => 'Notice updated successfully',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);
    }

    //for transactions
    public function transactionsIndex(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $transactions = SendmoneyOrder::all()->sortByDesc('created_at');
        return view('transactions.index', compact('dashboard', 'settings', 'transactions'));
    }

    public function transactionsDelete($id): RedirectResponse
    {
        $delete = SendmoneyOrder::findOrFail($id);
        $delete->delete();

        $notification = [
            'message' => 'Send money deleted successfully!',
            'alert-type' => 'success'
        ];
        return redirect()->back()->with($notification);
    }

    public function transactionsUpdate(Request $request)
    {
        $transaction = SendmoneyOrder::findOrFail($request->transaction_id);

        if (!$transaction) {
            return response()->json(['status' => false, 'message' => 'Transaction not found']);
        }

        if ($request->status == 'rejected') {
            $sender = User::findOrFail($transaction->user_id);
            if ($sender) {
                $sender->increment('balance', $transaction->amount);
            }
        }

        $transaction->status = $request->status;
        $transaction->save();

        return response()->json(['status' => true, 'message' => 'Transaction updated successfully']);
    }


    // for bulk sms configuration
    public function smsIndex(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $apiSms = SmsSetting::findOrFail(1);
        $permission = Permission::first();

        // Fetching Firebase configuration values from the config
        $firebaseApiKey = config('firebase.api_key');
        $firebaseAuthDomain = config('firebase.auth_domain');
        $firebaseProjectId = config('firebase.project_id');
        $firebaseMessagingSenderId = config('firebase.messaging_sender_id');
        $firebaseAppId = config('firebase.app_id');
        $firebaseMeasurementId = config('firebase.measurement_id');
        $firebase_vapi_key = config('firebase.vapi_key');

        $filePath = 'json/firebase_credentials.json';
        $fileExists = Storage::exists($filePath);
        $fileContents = $fileExists ? Storage::get($filePath) : null;

        // Passing all values to the view
        return view('support.sms_api', compact(
            'dashboard',
            'settings',
            'apiSms',
            'firebaseApiKey',
            'firebaseAuthDomain',
            'firebaseProjectId',
            'firebaseMessagingSenderId',
            'firebaseAppId',
            'firebaseMeasurementId',
            'firebase_vapi_key',
            'fileExists', 'fileContents', 'permission'
        ));
    }

    public function updateSms(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'api_key' => 'required',
            'sender_id' => 'required',
        ], [
            'api_key.required' => 'Enter BULKSMSBD API KEY.',
            'sender_id.required' => 'Enter BULKSMSBD Sender ID.',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        SmsSetting::findOrFail(1)->update([
            'api_key' => $request->api_key,
            'sender_id' => $request->sender_id
        ]);

        $notification = [
            'message' => 'Configuration updated successfully!',
            'alert-type' => 'success'
        ];
        return redirect()->back()->with($notification);

    }

    //================ Commission
    public function commissionIndex(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $commissions = Commission::where('type', 'commission')->get();
        $fees = Commission::where('type', 'fee')->get();
        return view('commission.index', compact('dashboard', 'settings', 'commissions', 'fees'));
    }

    public function updateCommission(Request $request, $id): RedirectResponse
    {

        $validator = Validator::make($request->all(), [
            'commission' => 'required|numeric|regex:/^\d+(\.\d{1,4})?$/',

        ]);
        if ($validator->fails()) {
            // Prepare a notification for feedback
            $notification = [
                'message' => $validator->errors()->first(),
                'alert-type' => 'error'
            ];

            // Redirect back with a success message
            return redirect()->back()->with($notification);
        }

        $commission = Commission::where('type', 'commission')->where('id', $id)->first();

        // Update commission amount and status based on the form input
        $commission->amount = $request->input('commission');
        $commission->status = $request->input('status') === 'on' ? 'active' : 'inactive';

        // Save the updated commission
        $commission->save();

        // Prepare a notification for feedback
        $notification = [
            'message' => $commission->name . ' commission updated successfully!',
            'alert-type' => 'success'
        ];

        // Redirect back with a success message
        return redirect()->back()->with($notification);
    }

    public function updateFees(Request $request, $id): RedirectResponse
    {

        $validator = Validator::make($request->all(), [
            'fee' => 'required|numeric|regex:/^\d+(\.\d{1,4})?$/',

        ]);
        if ($validator->fails()) {
            // Prepare a notification for feedback
            $notification = [
                'message' => $validator->errors()->first(),
                'alert-type' => 'error'
            ];

            // Redirect back with a success message
            return redirect()->back()->with($notification);
        }

        $fee = Commission::where('type', 'fee')->where('id', $id)->first();

        // Update commission amount and status based on the form input
        $fee->amount = $request->input('fee');
        $fee->status = $request->input('status') === 'on' ? 'active' : 'inactive';

        $fee->save();

        // Prepare a notification for feedback
        $notification = [
            'message' => $fee->name . ' fee updated successfully!',
            'alert-type' => 'success'
        ];

        // Redirect back with a success message
        return redirect()->back()->with($notification);
    }

    public function aboutCompanyIndex(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::first();
        $company = Company::first();
        return view('company.index', compact('dashboard', 'settings', 'company'));
    }

    public function aboutCompanyUpdate(Request $request): RedirectResponse
    {
        // Validate the incoming request
        $request->validate([
            'about' => 'nullable|string',
            'trade_license' => 'nullable|file|mimes:pdf,jpeg,png,jpg|max:2048',
            'certificate' => 'nullable|file|mimes:pdf,jpeg,png,jpg|max:2048',
        ]);

        $company = Company::first(); // Get the company record

        // Update 'about' section
        $company->about_company = $request->about;

        // Handle the Trade License upload
        if ($request->hasFile('trade_license')) {
            if ($company->trade_license && File::exists(public_path($company->trade_license))) {
                File::delete(public_path($company->trade_license));
            }
            $company->trade_license = $this->processImage($request->file('trade_license'), 'trade_license');
        }

        // Handle the Certificate upload
        if ($request->hasFile('certificate')) {
            if ($company->certificate && File::exists(public_path($company->certificate))) {
                File::delete(public_path($company->certificate));
            }
            $company->certificate = $this->processImage($request->file('certificate'), 'certificate');
        }

        // Save the company data
        $company->update();

        $notification = [
            'message' => 'Company updated successfully!',
            'alert-type' => 'success'
        ];
        return redirect()->back()->with($notification);
    }

    public function pdfUploadUpdate(Request $request): RedirectResponse
    {
        $request->validate([
            'reg_pdf' => 'required|file|mimes:pdf|max:2048',
        ]);

        $company = Company::first();

        if ($request->hasFile('reg_pdf')) {
            if ($company->reg_pdf && File::exists(public_path($company->reg_pdf))) {
                File::delete(public_path($company->reg_pdf));
            }
            $company->reg_pdf = $this->processImage($request->file('reg_pdf'), 'reg_pdf');
        }

        $company->update();

        $notification = [
            'message' => 'PDF Registration updated successfully.!',
            'alert-type' => 'success'
        ];
        return redirect()->back()->with($notification);
    }


    public function offerIndex(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::first();
        $offers = Offer::all();
        return view('offer.index', compact('dashboard', 'settings', 'offers'));
    }

    public function storeOffer(Request $request): RedirectResponse
    {
        $request->validate([
            'title' => 'required',
            'url' => 'required',
            'thumbnail' => 'required|max:2048',
        ]);

        $store = new Offer();
        $store->title = $request->input('title');
        $store->url = $request->input('url');
        $store->image = $this->processImage($request->file('thumbnail'), 'offer');
        $store->save();

        $notification = [
            'message' => $store->title . ' offer added successfully!', ' ',
            'alert-type' => 'success'
        ];
        return redirect()->back()->with($notification);
    }


    public function videoIndex(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::first();
        $videos = VideoTutorial::first();
        return view('videos.index', compact('dashboard', 'settings', 'videos'));
    }

    public function updateVideo(Request $request): RedirectResponse
    {
        $update = VideoTutorial::first();

        if ($request->filled('reg_video')) {
            $update->reg_video = $request->input('reg_video');
        }
        if ($request->filled('recharge_video')) {
            $update->recharge_video = $request->input('recharge_video');
        }
        if ($request->filled('send_money_video')) {
            $update->send_money_video = $request->input('send_money_video');
        }
        if ($request->filled('pay_bill_video')) {
            $update->pay_bill_video = $request->input('pay_bill_video');
        }
        if ($request->filled('add_money_video')) {
            $update->add_money_video = $request->input('add_money_video');
        }
        if ($request->filled('bank_transfer_video')) {
            $update->bank_transfer_video = $request->input('bank_transfer_video');
        }
        if ($request->filled('mobile_banking_video')) {
            $update->mobile_banking_video = $request->input('mobile_banking_video');
        }
        if ($request->filled('remittance_video')) {
            $update->remittance_video = $request->input('remittance_video');
        }
        $update->update();

        $notification = [
            'message' => 'Videos updated successfully!',
            'alert-type' => 'success'
        ];
        return redirect()->back()->with($notification);
    }

    public function deleteVideo($id): RedirectResponse
    {
        $delete = Videos::findOrFail($id);
        $delete->delete();
        $notification = [
            'message' => 'Video deleted successfully',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);
    }

    //todo: Permissions
    public function updatePermission(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'status' => 'required|in:active,inactive'
        ], [
            'status.required' => 'Status is required',
            'status.in' => 'Status must be active or inactive'
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $update = Permission::first();
        $update->status = $request->input('status');
        $update->update();

        $notification = [
            'message' => 'Status change to ' . $update->status,
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);
    }

}
