<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AboutSection;
use App\Models\Apk;
use App\Models\DownloadSection;
use App\Models\FeatureSection;
use App\Models\HeaderSection;
use App\Models\Screenshot;
use App\Models\ScreenshotSection;
use App\Models\Settings;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\View\View;
use Pion\Laravel\ChunkUpload\Handler\HandlerFactory;
use Pion\Laravel\ChunkUpload\Receiver\FileReceiver;
use Symfony\Component\HttpFoundation\BinaryFileResponse;

class FrontendController extends Controller
{
    public function hero(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::first();
        $headerSection = HeaderSection::first();
        return view('web_settings.hero_section.index', compact('dashboard', 'settings', 'headerSection'));
    }

    public function about(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::first();
        $aboutSection = AboutSection::first();
        return view('web_settings.about_section.index', compact('dashboard', 'settings', 'aboutSection'));
    }

    public function features(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::first();
        $featureSection = FeatureSection::first();
        return view('web_settings.feature_section.index', compact('dashboard', 'settings', 'featureSection'));
    }

    public function download(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::first();
        $downloadSection = DownloadSection::first();
        return view('web_settings.download_section.index', compact('dashboard', 'settings', 'downloadSection'));
    }

    public function screenshotsIndex(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::first();
        $ssSection = ScreenshotSection::first();
        $screenshots = Screenshot::orderBy('created_at', 'desc')->paginate(10);
        return view('web_settings.screenshot_section.index', compact('dashboard', 'settings', 'screenshots', 'ssSection'));
    }

    public function updateHero(Request $request): RedirectResponse
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required',
            'short_title' => 'required',
            'short_description' => 'required',
            't_client' => 'required',
            't_download' => 'required',
            't_rates' => 'required',
            't_win' => 'required',
            'logo' => 'required|image|mimes:jpeg,png,jpg,gif,svg|max:2048|dimensions:height=45, width=161',
            'h_image' => 'required|image|mimes:jpeg,png,jpg,gif,svg|max:2048|dimensions:height=1126, width=850',
            'h_bg_image' => 'required|image|mimes:jpeg,png,jpg,gif,svg|max:2048|dimensions:height=1080, width=1920',
        ], [
            'title.required' => 'Please enter title',
            'short_title.required' => 'Please enter short title',
            'short_description.required' => 'Please enter short description',
            't_client.required' => 'Please enter total clients',
            't_download.required' => 'Please enter total downloads',
            't_rates.required' => 'Please enter total rates',
            't_win.required' => 'Please enter total awards',
            'logo.required' => 'Please upload Logo',
            'logo.image' => 'Please upload Logo',
            'logo.mimes' => 'Please upload jpeg, jpg, gif and svg',
            'logo.dimensions' => 'Logo  Width x Height should be 161x45',
            'h_image.required' => 'Please upload image',
            'h_image.image' => 'Please upload image',
            'h_image.mimes' => 'Please upload jpeg, jpg, gif and svg',
            'h_image.dimensions' => 'Header Background Image Width x Height should be 850x1126',
            'h_bg_image.required' => 'Please upload image',
            'h_bg_image.image' => 'Please upload image',
            'h_bg_image.mimes' => 'Please upload jpeg, jpg, gif and svg',
            'h_bg_image.dimensions' => 'Screenshot Image Width x Height should be 1920x1080',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $update = HeaderSection::first();
        $update->title = $request->title;
        $update->short_title = $request->short_title;
        $update->total_clients = $request->t_client;
        $update->total_download = $request->t_download;
        $update->total_rates = $request->t_rates;
        $update->total_awards = $request->t_win;
        $update->short_description = $request->short_description;
        if ($request->hasFile('logo')) {
            if ($update->logo && File::exists(public_path($update->logo))) {
                File::delete(public_path($update->logo));
            }
            $update->logo = $this->processImage($request->file('logo'), 'site_logo');
        }

        if ($request->hasFile('h_image')) {
            if ($update->h_screenshot && File::exists(public_path($update->h_screenshot))) {
                File::delete(public_path($update->h_screenshot));
            }
            $update->h_screenshot = $this->processImage($request->file('h_image'), 'h_image');
        }

        if ($request->hasFile('h_bg_image')) {
            if ($update->h_bg_image && File::exists(public_path($update->h_bg_image))) {
                File::delete(public_path($update->h_bg_image));
            }
            $update->h_bg_image = $this->processImage($request->file('h_bg_image'), 'h_bg_image');
        }
        $update->update();

        $notification = [
            'message' => 'Hero Section Updated Successfully',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);
    }

    public function updateAbout(Request $request): RedirectResponse
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required',
            'short_title' => 'required',
            'description' => 'required',
            'about_img' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048|dimensions:height=150, width=140',
        ], [
            'title.required' => 'Please enter title',
            'short_title.required' => 'Please enter short title',
            'description.required' => 'Please enter description',
            'about_img.required' => 'Please upload image',
            'about_img.image' => 'Please upload image',
            'about_img.mimes' => 'Please upload jpeg, jpg, gif and svg',
            'about_img.dimensions' => 'About Image Width x Height should be 140x150',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $update = AboutSection::first();
        $update->title = $request->title;
        $update->short_title = $request->short_title;
        $update->description = $request->description;
        if ($request->hasFile('about_img')) {
            if ($update->about_img && File::exists(public_path($update->about_img))) {
                File::delete(public_path($update->about_img));
            }
            $update->about_img = $this->processImage($request->file('about_img'), 'about_img');
        }
        $update->update();

        $notification = [
            'message' => 'About Section Updated Successfully',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);
    }

    public function uploadScreenshot(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'images' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048|dimensions:height=1200, width=540',
        ], [
            'images.required' => 'Please upload image',
            'images.image' => 'Please upload image',
            'images.mimes' => 'Please upload jpeg, jpg, gif and svg',
            'images.dimensions' => 'Screenshot Width x Height should be 1200x540',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $store = new Screenshot();
        if ($request->hasFile('images')) {
            $store->images = $this->processImage($request->file('images'), 'screenshots');

        }
        $store->save();

        $notification = [
            'message' => 'Screenshot Upload Successfully',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);
    }

    public function screenshotDetailsUpdate(Request $request): RedirectResponse
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required',
            'short_title' => 'required',
            'description' => 'required',
        ], [
            'title.required' => 'Please enter title',
            'short_title.required' => 'Please enter short title',
            'description.required' => 'Please enter description',
        ]);

        if ($validator->fails()) {
            Log::error("error", [$validator->errors()]);
            return back()->withErrors($validator)->withInput();
        }

        $update = ScreenshotSection::first();
        $update->title = $request->title;
        $update->short_title = $request->short_title;
        $update->description = $request->description;
        $update->update();

        $notification = [
            'message' => 'Screenshot Section Updated Successfully',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);
    }

    public function updateFeature(Request $request): RedirectResponse
    {
        Log::info('requested', [$request->all()]);
        $validator = Validator::make($request->all(), [
            'title' => 'required',
            'short_title' => 'required',
            'description' => 'required',
            't_one' => 'required',
            't_two' => 'required',
            't_three' => 'required',
            't_four' => 'required',
            't_five' => 'required',
            't_six' => 'required',
            'd_one' => 'required',
            'd_two' => 'required',
            'd_three' => 'required',
            'd_four' => 'required',
            'd_five' => 'required',
            'd_six' => 'required',
            'f_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048|dimensions:height=595, width=375',
        ], [
            'title.required' => 'Please enter title',
            'short_title.required' => 'Please enter short title',
            'description.required' => 'Please enter description',
            't_one.required' => 'Please enter title one',
            't_two.required' => 'Please enter title two',
            't_three.required' => 'Please enter title three',
            't_four.required' => 'Please enter title four',
            't_five.required' => 'Please enter title six',
            't_six.required' => 'Please enter title seven',
            'd_one.required' => 'Please enter description one',
            'd_two.required' => 'Please enter description two',
            'd_three.required' => 'Please enter description three',
            'd_four.required' => 'Please enter description four',
            'd_five.required' => 'Please enter description five',
            'd_six.required' => 'Please enter description six',
            'f_image.required' => 'Please upload image',
            'f_image.image' => 'Please upload image',
            'f_image.mimes' => 'Please upload jpeg, jpg, gif and svg',
            'f_image.dimensions' => 'About Image Width x Height should be 140x150',
        ]);

        if ($validator->fails()) {
            Log::error("error", [$validator->errors()]);
            return back()->withErrors($validator)->withInput();
        }

        $update = FeatureSection::first();
        $update->title = $request->title;
        $update->short_title = $request->short_title;
        $update->t_one = $request->t_one;
        $update->t_two = $request->t_two;
        $update->t_three = $request->t_three;
        $update->t_four = $request->t_four;
        $update->t_five = $request->t_five;
        $update->t_six = $request->t_six;
        $update->d_one = $request->d_one;
        $update->d_two = $request->d_two;
        $update->d_three = $request->d_three;
        $update->d_four = $request->d_four;
        $update->d_five = $request->d_five;
        $update->d_six = $request->d_six;
        $update->description = $request->description;
        if ($request->hasFile('f_image')) {
            if ($update->f_image && File::exists(public_path($update->f_image))) {
                File::delete(public_path($update->f_image));
            }
            $update->f_image = $this->processImage($request->file('f_image'), 'f_image');
        }
        $update->update();

        $notification = [
            'message' => 'Feature Section Updated Successfully',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);
    }

    public function updateVideo(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'url' => 'required|url',
        ], [
            'url.required' => 'Please enter url',
            'url.url' => 'Please enter url',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $update = DownloadSection::first();
        $update->video_url = $request->url;
        $update->update();

        $notification = [
            'message' => 'Video Section Updated Successfully',
            'alert-type' => 'success'
        ];
        return redirect()->back()->with($notification);
    }

    private function processImage($file, $fieldName): string
    {
        $fileName = time() . '_' . $fieldName . '.' . $file->getClientOriginalExtension();
        $path = 'upload/admins/' . $fileName;
        $file->move(public_path('upload/admins/'), $fileName);
        return $path;
    }

    public function uploadApk(Request $request): JsonResponse
    {
        try {
            $receiver = new FileReceiver("file", $request, HandlerFactory::classFromRequest($request));

            if (!$receiver->isUploaded()) {
                return response()->json([
                    'status' => false,
                    'error' => 'No file uploaded.'
                ], 400);
            }

            $save = $receiver->receive();

            if ($save->isFinished()) {
                $file = $save->getFile();

                $extension = strtolower($file->getClientOriginalExtension());
                if ($extension !== 'apk') {
                    return response()->json([
                        'status' => false,
                        'error' => 'Only APK files are allowed.'
                    ], 400);
                }

                $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                $sanitizedFileName = preg_replace('/[^A-Za-z0-9_\-]/', '_', $originalName);
                $fileName = $sanitizedFileName . '.apk';

                $file->move(public_path('apk'), $fileName);

                $store = DownloadSection::first() ?? new DownloadSection();

                if ($store->app_path && File::exists(public_path($store->app_path))) {
                    File::delete(public_path($store->app_path));
                }

                $store->app_path = 'apk/' . $fileName;
                $store->app_name = $sanitizedFileName;
                $store->save();

                return response()->json([
                    'status' => true,
                    'message' => 'APK uploaded successfully!',
                    'file' => $fileName,
                    'db_path' => $store->file_path
                ]);
            }

            $handler = $save->handler();
            return response()->json([
                "status" => true,
                "done" => $handler->getPercentageDone(),
                "chunks" => $handler->getTotalChunks(),
                "current" => $handler->getCurrentChunk(),
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'error' => 'Server error: ' . $e->getMessage()
            ], 500);
        }
    }

    public function downloadApk(): BinaryFileResponse|RedirectResponse
    {
        $apk = DownloadSection::first();

        if ($apk) {
            $path = public_path($apk->app_path);

            if (File::exists($path)) {
                return response()->download($path, $apk->app_name . '.apk', [
                    'Content-Type' => 'application/vnd.android.package-archive',
                    'Content-Disposition' => 'attachment; filename="' . $apk->app_path . '.apk"',
                    'X-Content-Type-Options' => 'nosniff'
                ]);
            }
        }

        return back()->with('error', 'APK file not found.');
    }

}
