<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Controllers\FcmController;
use App\Models\AddMoneyOrder;
use App\Models\AutoPay;
use App\Models\BankGateway;
use App\Models\MobileGateway;
use App\Models\Settings;
use App\Models\User;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\View\View;

class GatewayController extends Controller
{
    public function index(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $m_gateways = MobileGateway::all()->sortByDesc('created_at');
        $b_gateways = BankGateway::all()->sortByDesc('created_at');
        return view('gateway.index', compact('dashboard', 'settings', 'm_gateways', 'b_gateways'));
    }

    public function addGateway(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $gateway = AutoPay::findOrFail(1);
        return view('gateway.add', compact('dashboard', 'settings', 'gateway'));
    }

    public function storeBank(Request $request): RedirectResponse
    {
        $validator = Validator::make(request()->all(), [
            'name' => 'required',
            'ac_holder' => 'required',
            'ac_number' => 'required',
            'ac_ibn' => 'required',
            'branch_name' => 'required',
            'country' => 'required',
            'logo' => 'required|image|mimes:jpeg,png,jpg,|max:2048',
        ], [
            'name.required' => 'Please enter bank name',
            'ac_holder.required' => 'Please enter account holder name',
            'ac_number.required' => 'Please enter account number',
            'ac_ibn.required' => 'Please enter account IBN',
            'branch_name.required' => 'Please enter branch name',
            'country.required' => 'Please enter country',
            'logo.required' => 'Please upload logo',
            'logo.image' => 'The logo must be an image',
            'logo.mimes' => 'Please upload logo jpg, jpeg, png',
            'logo.max' => 'Max size 2MB',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $store = new BankGateway();
        $store->bank_name = $request->name;
        $store->slug = Str::slug($request->name);
        $store->ac_holder_name = $request->ac_holder;
        $store->ac_holder_number = $request->ac_number;
        $store->ibn_no = $request->ac_ibn;
        $store->branch_name = $request->branch_name;
        $store->country_name = $request->country;

        if ($request->hasFile('logo')) {
            $file = $request->file('logo');
            $fileName = time() . '.' . $file->getClientOriginalExtension();
            $path = 'upload/gateway/' . $fileName;
            $file->move(public_path('upload/gateway/'), $fileName);
            $store->logo = $path; // Store the file path in the database
        }

        $status = $request->has('status') == 'on' ? true : false;
        $store->status = $status;
        $store->save();

        $notification = [
            'message' => $request->name . ' added successfully',
            'alert-type' => 'success',
        ];

        return back()->with($notification);
    }

    public function storeMobile(Request $request)
    {
        $validator = Validator::make(request()->all(), [
            'm_name' => 'required',
            'm_number' => 'required',
            'type' => 'required',
            'm_logo' => 'required|image|mimes:jpeg,png,jpg,|max:2048',
        ], [
            'm_name.required' => 'Please enter mobile banking name',
            'm_number.required' => 'Please enter mobile banking number',
            'type.required' => 'Please enter add money type',
            'm_logo.required' => 'Please upload logo',
            'm_logo.image' => 'The logo must be an image',
            'm_logo.mimes' => 'Please upload logo jpg, jpeg, png',
            'm_logo.max' => 'Max size 2MB',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        // Generate the initial slug from the name
        $slug = Str::slug($request->m_name);

        // Check if slug exists and append type if not unique
        $existingSlug = MobileGateway::where('slug', $slug)->exists();
        if ($existingSlug) {
            $slug = $slug . '-' . Str::slug($request->type);
        }

        $store = new MobileGateway();
        $store->name = $request->m_name;
        $store->slug = $slug;
        $store->number = $request->m_number;
        $store->sending_type = $request->type;

        if ($request->hasFile('m_logo')) {
            $file = $request->file('m_logo');
            $fileName = time() . '.' . $file->getClientOriginalExtension();
            $path = 'upload/gateway/' . $fileName;
            $file->move(public_path('upload/gateway/'), $fileName);
            $store->logo = $path; // Store the file path in the database
        }

        $status = $request->has('status') == 'on' ? true : false;
        $store->status = $status;
        $store->save();

        $notification = [
            'message' => $request->m_name . ' added successfully',
            'alert-type' => 'success',
        ];

        return back()->with($notification);
    }

    public function updateAutoGateWay(Request $request): RedirectResponse
    {
        $update = AutoPay::findOrFail(1);
        $update->api_key = $request->brand_key;
        $update->update();

        $notification = [
            'message' => 'AutoPay update successfully',
            'alert-type' => 'success',
        ];

        return back()->with($notification);
    }

    public function editBank($id): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $bank = BankGateway::findOrFail($id);
        return view('gateway.edit_bank', compact('dashboard', 'settings', 'bank'));
    }

    public function editMobile($id): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $mobile = MobileGateway::findOrFail($id);
        return view('gateway.edit_mobile', compact('dashboard', 'settings', 'mobile'));
    }

    public function updateBank(Request $request, $id)
    {
        $validator = Validator::make(request()->all(), [
            'name' => 'required',
            'ac_holder' => 'required',
            'ac_number' => 'required',
            'ac_ibn' => 'required',
            'branch_name' => 'required',
            'country' => 'required',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,|max:2048',
        ], [
            'name.required' => 'Please enter bank name',
            'ac_holder.required' => 'Please enter account holder name',
            'ac_number.required' => 'Please enter account number',
            'ac_ibn.required' => 'Please enter account IBN',
            'branch_name.required' => 'Please enter branch name',
            'country.required' => 'Please enter country',
            'logo.image' => 'The logo must be an image',
            'logo.mimes' => 'Please upload logo jpg, jpeg, png',
            'logo.max' => 'Max size 2MB',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $update = BankGateway::findOrFail($id);
        $update->bank_name = $request->name;
        $update->slug = Str::slug($request->name);
        $update->ac_holder_name = $request->ac_holder;
        $update->ac_holder_number = $request->ac_number;
        $update->ibn_no = $request->ac_ibn;
        $update->branch_name = $request->branch_name;
        $update->country_name = $request->country;

        if ($request->hasFile('logo')) {

            if ($update->logo && File::exists(public_path($update->logo))) {
                File::delete(public_path($update->logo));
            }

            $file = $request->file('logo');
            $fileName = time() . '.' . $file->getClientOriginalExtension();
            $path = 'upload/gateway/' . $fileName;
            $file->move(public_path('upload/gateway/'), $fileName);
            $update->logo = $path; // Store the file path in the database
        }

        $status = $request->has('status') == 'on' ? true : false;
        $update->status = $status;
        $update->save();

        $notification = [
            'message' => $request->name . ' update successfully',
            'alert-type' => 'success',
        ];

        return back()->with($notification);
    }

    public function updateMobile(Request $request, $id)
    {
        $validator = Validator::make(request()->all(), [
            'm_name' => 'required',
            'm_number' => 'required',
            'type' => 'required',
            'm_logo' => 'nullable|image|mimes:jpeg,png,jpg,|max:2048',
        ], [
            'm_name.required' => 'Please enter mobile banking name',
            'm_number.required' => 'Please enter mobile banking number',
            'type.required' => 'Please enter add money type',
            'm_logo.image' => 'The logo must be an image',
            'm_logo.mimes' => 'Please upload logo jpg, jpeg, png',
            'm_logo.max' => 'Max size 2MB',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $update = MobileGateway::findOrFail($id);
        $update->name = $request->m_name;
        $update->slug = Str::slug($request->m_name);
        $update->number = $request->m_number;
        $update->sending_type = $request->type;

        if ($request->hasFile('m_logo')) {

            if ($update->logo && File::exists(public_path($update->logo))) {
                File::delete(public_path($update->logo));
            }

            $file = $request->file('m_logo');
            $fileName = time() . '.' . $file->getClientOriginalExtension();
            $path = 'upload/gateway/' . $fileName;
            $file->move(public_path('upload/gateway/'), $fileName);
            $update->logo = $path; // Store the file path in the database
        }

        $status = $request->has('status') == 'on' ? true : false;
        $update->status = $status;
        $update->save();

        $notification = [
            'message' => $request->m_name . ' update successfully',
            'alert-type' => 'success',
        ];

        return back()->with($notification);
    }

    public function deleteMobile($id): RedirectResponse
    {
        $delete = MobileGateway::findOrFail($id);
        if ($delete->logo && File::exists(public_path($delete->logo))) {
            File::delete(public_path($delete->logo));
        }
        $delete->delete();
        $notification = [
            'message' => $delete->name . ' deleted successfully',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);
    }

    public function deleteBank($id): RedirectResponse
    {
        $delete = BankGateway::findOrFail($id);
        if ($delete->logo && File::exists(public_path($delete->logo))) {
            File::delete(public_path($delete->logo));
        }
        $delete->delete();
        $notification = [
            'message' => $delete->name . ' deleted successfully',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);
    }

    //============= Add Money actions
    public function addMoneyPending(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $items = AddMoneyOrder::where('status', 'pending')->latest()->get();
        return view('gateway.add_money_order', compact('dashboard', 'settings', 'items'));
    }

    public function addMoneySuccess(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $items = AddMoneyOrder::where('status', 'success')->latest()->get();
        return view('gateway.add_money_order', compact('dashboard', 'settings', 'items'));
    }

    public function addMoneyRejected(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $items = AddMoneyOrder::where('status', 'cancelled')->latest()->get();
        return view('gateway.add_money_order', compact('dashboard', 'settings', 'items'));
    }

    public function addMoneyApproved(Request $request): RedirectResponse
    {
        $notificationController = new FcmController();

        $update = AddMoneyOrder::where('id', $request->success_id)->first();
        $update->update([
            'status' => 'success'
        ]);

        $update->user->balance += $update->amount;
        $update->user->update();

        $notificationController->sendFcmNotification($update->user_id, 'Add Money', 'Your ' . $update->method_name . ' addmoney is approved');

        $notification = [
            'message' => 'This ' . $update->method_name . ' service is approved successfully',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);

    }

    public function addMoneyOrderRejected(Request $request): RedirectResponse
    {
        $notificationController = new FcmController();

        $validator = Validator::make($request->all(), [
            'reason' => 'required'
        ], [
            'reason.required' => 'Write Rejected Reason',
        ]);

        if ($validator->fails()) {

            $notification = [
                'message' => $validator->errors()->first(),
                'alert-type' => 'error'
            ];

            return redirect()->back()->with($notification);
        }

        $update = AddMoneyOrder::where('id', $request->reject_id)->first();
        $update->update([
            'status' => 'cancelled',
            'reason' => $request->reason
        ]);


        $notificationController->sendFcmNotification($update->user_id, 'Add Money', 'Your ' . $update->method_name . ' addmoney is rejected');


        $notification = [
            'message' => 'This ' . $update->method_name . ' service is rejected successfully!',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);

    }

    public function addMoneyDelete($id): RedirectResponse
    {
        $delete = AddMoneyOrder::findOrFail($id);
        $delete->delete();

        $notification = [
            'message' => 'This service is deleted successfully!',
            'alert-type' => 'success'
        ];
        return redirect()->back()->with($notification);
    }

}
