<?php

namespace App\Http\Controllers\Admin;

use App\Events\MessageSent;
use App\Http\Controllers\Controller;
use App\Http\Controllers\FcmController;
use App\Models\Chat;
use App\Models\Settings;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\View\View;

class LiveChatController extends Controller
{
    public function index(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);

        // চেক করুন যদি ইউজার 'admin' হয়, তাহলে সব ইউজার শো হবে
        if (auth()->user()->role == 'admin') {
            // অ্যাডমিনের জন্য, শুধু সেই ইউজারদের দেখাবো যারা মেসেজ পাঠিয়েছে
            $usersWithMessages = User::whereHas('chats', function ($query) {
                $query->where('receiver_id', auth()->id());  // যেখানে অ্যাডমিন রিসিভার
            })->get();

            // Get unread message count for each user
            foreach ($usersWithMessages as $user) {
                $user->unread_message_count = Chat::where('receiver_id', auth()->id())
                    ->where('sender_id', $user->id)
                    ->where('is_closed', false)
                    ->count();
            }

            return view('live_chat.index', compact('usersWithMessages', 'dashboard', 'settings'));
        } else {
            // ইউজারের জন্য, সব অ্যাডমিন শো হবে
            $admins = User::where('role', 'admin')->get();
            return view('live_chat.index', compact('admins', 'dashboard', 'settings'));
        }
    }

    public function getMessages($userId): JsonResponse
    {
        $messages = Chat::where(function ($query) use ($userId) {
            $query->where('sender_id', Auth::id())->where('receiver_id', $userId);
        })->orWhere(function ($query) use ($userId) {
            $query->where('sender_id', $userId)->where('receiver_id', Auth::id());
        })->orderBy('created_at', 'asc')->get();

        Chat::where('receiver_id', Auth::id())
            ->where('sender_id', $userId)
            ->update(['is_closed' => true]);

        return response()->json(['messages' => $messages]);
    }

    public function sendMessage(Request $request): JsonResponse
    {
        $notificationController = new FcmController();
        try {
            $request->validate([
                'message' => 'required|string',
                'receiver_id' => 'required|exists:users,id',
            ]);

            $message = Chat::create([
                'sender_id' => Auth::id(),
                'receiver_id' => $request->receiver_id,
                'message' => $request->message,
            ]);

            broadcast(new MessageSent($message->message, auth()->user()->id, $request->receiver_id, \Carbon\Carbon::parse($message->created_at)->toIso8601String()));
            $notificationController->sendFcmNotification($request->receiver_id, 'Support Message', $request->message);

            return response()->json([
                'success' => true,
                'message' => 'Message sent successfully!'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to send message'
            ], 500);
        }
    }
}
