<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\FcmController;
use App\Http\Controllers\Controller;
use App\Models\DriveOrder;
use App\Models\Package;
use App\Models\Settings;
use App\Models\User;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\View\View;

class PackageController extends Controller
{

    //==============>> Drive Package <<==============//
    public function drivePackageIndex(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $packages = Package::where('package', 'Drive')->get()->sortByDesc('created_at');
        return view('drives.index', compact('dashboard', 'settings', 'packages'));
    }

    public function addDrivePackage(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        return view('drives.add', compact('dashboard', 'settings'));
    }

    public function storeDrivePackage(Request $request): RedirectResponse
    {
        $validate = Validator::make(request()->all(), [
            'name' => 'required',
            'buying_price' => 'required',
            'selling_price' => 'required',
            'discount_price' => 'required',
            'operator' => 'required',
            'validity' => 'required',
        ], [
            'name.required' => 'Offer name is required',
            'buying_price.required' => 'Buying price is required',
            'selling_price.required' => 'Selling price is required',
            'discount_price.required' => 'Discount price is required',
            'operator.required' => 'Select an operator Exp: Grameenphone, Robi, Banglalink, Airtel, Teletalk Or Skitto',
            'validity.required' => 'Enter validity date',
        ]);

        if ($validate->fails()) {
            return back()->withErrors($validate)->withInput();
        }

        $store = new Package();
        $store->name = $request->name;
        $store->slug = $this->createUniqueSlug($request->name);
        $store->buying_price = $request->buying_price;
        $store->selling_price = $request->selling_price;
        $store->discount_price = $request->discount_price;
        $store->operator = $request->operator;
        $store->package = 'Drive';
        $store->validity = $request->validity;
        $store->save();

        $notification = [
            'message' => $request->name . ' added successfully',
            'alert-type' => 'success',
        ];

        return back()->with($notification);
    }

    public function editDrivePackage($id): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $package = Package::findOrFail($id);
        return view('drives.edit', compact('dashboard', 'settings', 'package'));
    }

    public function updateDrivePackage(Request $request, $slug): RedirectResponse
    {
        $validate = Validator::make(request()->all(), [
            'name' => 'required',
            'buying_price' => 'required',
            'selling_price' => 'required',
            'discount_price' => 'required',
            'operator' => 'required',
            'validity' => 'required',
        ], [
            'name.required' => 'Offer name is required',
            'buying_price.required' => 'Buying price is required',
            'selling_price.required' => 'Selling price is required',
            'discount_price.required' => 'Discount price is required',
            'operator.required' => 'Select an operator Exp: Grameenphone, Robi, Banglalink, Airtel, Teletalk',
            'validity.required' => 'Enter validity date',
        ]);

        if ($validate->fails()) {
            return back()->withErrors($validate)->withInput();
        }

        $update_data = Package::where('slug', $slug)->first();
        $update_data->name = $request->name;
        $update_data->buying_price = $request->buying_price;
        $update_data->selling_price = $request->selling_price;
        $update_data->discount_price = $request->discount_price;
        $update_data->operator = $request->operator;
        $update_data->package = 'Drive';
        $update_data->validity = $request->validity;
        $update_data->update();

        $notification = [
            'message' => $request->name . ' added successfully',
            'alert-type' => 'success',
        ];

        return back()->with($notification);
    }

    public function deleteDrivePackage($id): RedirectResponse
    {
        $delete = Package::findOrFail($id);
        $delete->delete();
        $notification = [
            'message' => $delete->name . ' deleted successfully',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);
    }

    public function updateStatus(Request $request)
    {
        $request->validate([
            'package_ids' => 'required|array',
            'package_ids.*' => 'exists:packages,id',
            'status' => 'required|in:active,deactive',
        ]);

        DB::beginTransaction();

        try {
            Package::whereIn('id', $request->package_ids)
                ->update(['status' => $request->status]);

            DB::commit();

            return response()->json(['status' => true, 'message' => 'Packages updated successfully']);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['status' => false, 'message' => 'Failed to update packages']);
        }
    }


    //============>> Regular Package <<=============//

    public function regularPackageIndex(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $packages = Package::where('package', 'Regular')->get()->sortByDesc('created_at');
        return view('regular.index', compact('dashboard', 'settings', 'packages'));
    }

    public function addRegularPackage(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        return view('regular.add', compact('dashboard', 'settings'));
    }

    public function storeRegularPackage(Request $request): RedirectResponse
    {
        $validate = Validator::make(request()->all(), [
            'name' => 'required',
            'buying_price' => 'required',
            'selling_price' => 'required',
            'discount_price' => 'required',
            'operator' => 'required',
            'validity' => 'required',
        ], [
            'name.required' => 'Offer name is required',
            'buying_price.required' => 'Buying price is required',
            'selling_price.required' => 'Selling price is required',
            'discount_price.required' => 'Discount price is required',
            'operator.required' => 'Select an operator Exp: Grameenphone, Robi, Banglalink, Airtel, Teletalk',
            'validity.required' => 'Enter validity date',
        ]);

        if ($validate->fails()) {
            return back()->withErrors($validate)->withInput();
        }

        $store = new Package();
        $store->name = $request->name;
        $store->slug = $this->createUniqueSlug($request->name);
        $store->buying_price = $request->buying_price;
        $store->selling_price = $request->selling_price;
        $store->discount_price = $request->discount_price;
        $store->operator = $request->operator;
        $store->package = 'Regular';
        $store->validity = $request->validity;
        $store->save();

        $notification = [
            'message' => $request->name . ' added successfully',
            'alert-type' => 'success',
        ];

        return back()->with($notification);
    }

    public function editRegularPackage($id): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $package = Package::findOrFail($id);
        return view('regular.edit', compact('dashboard', 'settings', 'package'));
    }

    public function updateRegularPackage(Request $request, $slug): RedirectResponse
    {
        $validate = Validator::make(request()->all(), [
            'name' => 'required',
            'buying_price' => 'required',
            'selling_price' => 'required',
            'discount_price' => 'required',
            'operator' => 'required',
            'validity' => 'required',
        ], [
            'name.required' => 'Offer name is required',
            'buying_price.required' => 'Buying price is required',
            'selling_price.required' => 'Selling price is required',
            'discount_price.required' => 'Discount price is required',
            'operator.required' => 'Select an operator Exp: Grameenphone, Robi, Banglalink, Airtel, Teletalk',
            'validity.required' => 'Enter validity date',
        ]);

        if ($validate->fails()) {
            return back()->withErrors($validate)->withInput();
        }

        $update_data = Package::where('slug', $slug)->first();
        $update_data->name = $request->name;
        $update_data->buying_price = $request->buying_price;
        $update_data->selling_price = $request->selling_price;
        $update_data->discount_price = $request->discount_price;
        $update_data->operator = $request->operator;
        $update_data->package = 'Regular';
        $update_data->validity = $request->validity;
        $update_data->update();

        $notification = [
            'message' => $request->name . ' added successfully',
            'alert-type' => 'success',
        ];

        return back()->with($notification);
    }

    public function deleteRegularPackage($id): RedirectResponse
    {
        $delete = Package::findOrFail($id);
        $delete->delete();
        $notification = [
            'message' => $delete->name . ' deleted successfully',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);
    }

    private function createUniqueSlug($name): string
    {
        // Generate the initial slug from the package name
        $slug = Str::slug($name);

        // Find existing slugs in the database that match or start with the generated slug
        $count = Package::where('slug', 'LIKE', "{$slug}%")->count();

        // If no existing slug is found, return the generated slug
        if ($count === 0) {
            return $slug;
        }

        // Append a number to the slug to make it unique
        return $slug . '-' . ($count + 1);
    }

    // for drive actions
    public function drivePackagePending(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $drives = DriveOrder::where('status', 'pending')->where('type', 'Drive')->latest()->get();
        return view('drives.package_drive_order', compact('dashboard', 'settings', 'drives'));
    }

    public function drivePackageSuccess(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $drives = DriveOrder::where('status', 'accepted')->where('type', 'Drive')->latest()->get();
        return view('drives.package_drive_order', compact('dashboard', 'settings', 'drives'));
    }

    public function drivePackageRejected(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $drives = DriveOrder::where('status', 'cancelled')->where('type', 'Drive')->latest()->get();
        return view('drives.package_drive_order', compact('dashboard', 'settings', 'drives'));
    }

    public function driveApproved(Request $request): RedirectResponse
    {
        $notificationController = new FcmController();

        $update = DriveOrder::where('id', $request->success_id)->where('type', 'Drive')->first();
        $update->update([
            'status' => 'accepted'
        ]);

        $notificationController->sendFcmNotification($update->user_id, 'Drive Order', 'Your drive order is approved');

        $notification = [
            'message' => 'This drive is approved successfully',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);

    }

    public function driveRejected(Request $request): RedirectResponse
    {
        $notificationController = new FcmController();


        $validator = Validator::make($request->all(), [
            'reason' => 'required'
        ], [
            'reason.required' => 'Write Drive Rejected Reason',
        ]);

        if ($validator->fails()) {

            $notification = [
                'message' => $validator->errors()->first(),
                'alert-type' => 'error'
            ];

            return redirect()->back()->with($notification);
        }

        $update = DriveOrder::where('id', $request->reject_id)->where('type', 'Drive')->first();
        $update->update([
            'status' => 'cancelled',
            'reason' => $request->reason
        ]);

        $update->user->balance += $update->pay;
        $update->user->update();

        $notificationController->sendFcmNotification($update->user_id, 'Drive Order', 'Your drive order is cancelled & refunded');

        $notification = [
            'message' => 'This drive is rejected successfully!',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);

    }

    public function driveDelete($id): RedirectResponse
    {
        $delete = DriveOrder::findOrFail($id);
        $delete->delete();

        $notification = [
            'message' => 'This drive offer is deleted successfully!',
            'alert-type' => 'success'
        ];
        return redirect()->back()->with($notification);
    }


    // for Regular package actions
    public function regularPackagePending(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $regular = DriveOrder::where('status', 'pending')->where('type', 'Regular')->latest()->get();
        return view('regular.package_regular_order', compact('dashboard', 'settings', 'regular'));
    }

    public function regularPackageSuccess(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $regular = DriveOrder::where('status', 'accepted')->where('type', 'Regular')->latest()->get();
        return view('regular.package_regular_order', compact('dashboard', 'settings', 'regular'));
    }

    public function regularPackageRejected(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $regular = DriveOrder::where('status', 'cancelled')->where('type', 'Regular')->latest()->get();
        return view('regular.package_regular_order', compact('dashboard', 'settings', 'regular'));
    }

    public function regularApproved(Request $request): RedirectResponse
    {
        $notificationController = new FcmController();


        $update = DriveOrder::where('id', $request->success_id)->where('type', 'Regular')->first();
        $update->update([
            'status' => 'accepted'
        ]);

        $notificationController->sendFcmNotification($update->user_id, 'Regular Offer', 'Your regular order is approved');

        $notification = [
            'message' => 'This Regular package is approved successfully',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);

    }

    public function regularRejected(Request $request): RedirectResponse
    {
        $notificationController = new FcmController();

        $validator = Validator::make($request->all(), [
            'reason' => 'required'
        ], [
            'reason.required' => 'Write Regular Package Rejected Reason',
        ]);

        if ($validator->fails()) {

            $notification = [
                'message' => $validator->errors()->first(),
                'alert-type' => 'error'
            ];

            return redirect()->back()->with($notification);
        }

        $update = DriveOrder::where('id', $request->reject_id)->where('type', 'Regular')->first();
        $update->update([
            'status' => 'cancelled',
            'reason' => $request->reason
        ]);

        $update->user->balance += $update->pay;
        $update->user->update();

        $notificationController->sendFcmNotification($update->user_id, 'Regular Offer', 'Your regular order is cancelled & refunded');

        $notification = [
            'message' => 'This regular package is rejected successfully!',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);

    }

    public function regularDelete($id): RedirectResponse
    {
        $delete = DriveOrder::findOrFail($id);
        $delete->delete();

        $notification = [
            'message' => 'This regular offer is deleted successfully!',
            'alert-type' => 'success'
        ];
        return redirect()->back()->with($notification);
    }


}
