<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Controllers\FcmController;
use App\Models\Remittance;
use App\Models\RemittanceOrder;
use App\Models\Settings;
use App\Models\User;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Illuminate\View\View;

class RemittanceController extends Controller
{
    public function index(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $remits = Remittance::all()->sortByDesc('created_at');
        return view('remittance.index', compact('dashboard', 'settings', 'remits'));
    }

    public function addRemitProvider(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        return view('remittance.add', compact('dashboard', 'settings'));
    }

    public function storeRemitProvider(Request $request): RedirectResponse
    {
        $validator = Validator::make(request()->all(), [
            'name' => 'required|unique:remittances,name',
            'logo' => 'required|image|mimes:jpeg,png,jpg,|max:2048',
        ], [
            'name.required' => 'Please enter Remittance provider name',
            'name.unique' => 'Remittance provider name already exists',
            'logo.required' => 'Please enter Remittance provider logo',
            'logo.image' => 'Please upload logo',
            'logo.mimes' => 'Please upload JPG, PNG, JPEG',
            'logo.max' => 'Maximum upload file size 2MB',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $store = new Remittance();
        $store->name = $request->name;
        $store->slug = Str::slug($request->name);

        if ($request->hasFile('logo')) {
            $file = $request->file('logo');
            $fileName = time() . '.' . $file->getClientOriginalExtension();
            $path = 'upload/remit/' . $fileName;
            $file->move(public_path('upload/remit/'), $fileName);
            $store->logo = $path; // Store the file path in the database
        }

        $status = $request->has('status') == 'on' ? true : false;
        $store->status = $status;
        $store->save();

        $notification = [
            'message' => $request->name . ' added successfully',
            'alert-type' => 'success',
        ];

        return back()->with($notification);
    }

    public function editRemitProvider($id): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $remit = Remittance::findOrFail($id);
        return view('remittance.edit', compact('dashboard', 'settings', 'remit'));
    }

    public function updateRemitProvider(Request $request, $id): RedirectResponse
    {
        $validator = Validator::make(request()->all(), [
            'name' => ['nullable', Rule::unique('remittances')->ignore($id)],
            'slug' => ['nullable', Rule::unique('remittances')->ignore($id)],
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,|max:2048',
        ], [
            'name.required' => 'Please enter Remittance provider name',
            'name.unique' => 'Remittance provider name already exists',
            'logo.required' => 'Please enter Remittance provider logo',
            'logo.image' => 'Please upload logo',
            'logo.mimes' => 'Please upload JPG, PNG, JPEG',
            'logo.max' => 'Maximum upload file size 2MB',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $update = Remittance::findOrFail($id);
        $update->name = $request->name;
        $update->slug = Str::slug($request->name);

        if ($request->hasFile('logo')) {

            if ($update->logo && File::exists(public_path($update->logo))) {
                File::delete(public_path($update->logo));
            }

            $file = $request->file('logo');
            $fileName = time() . '.' . $file->getClientOriginalExtension();
            $path = 'upload/remit/' . $fileName;
            $file->move(public_path('upload/remit/'), $fileName);
            $update->logo = $path; // Store the file path in the database
        }

        $status = $request->has('status') == 'on' ? true : false;
        $update->status = $status;
        $update->update();

        $notification = [
            'message' => $request->name . ' updated successfully',
            'alert-type' => 'success',
        ];

        return back()->with($notification);
    }

    public function deleteRemitProvider($id): RedirectResponse
    {
        $delete = Remittance::findOrFail($id);
        if ($delete->logo && File::exists(public_path($delete->logo))) {
            File::delete(public_path($delete->logo));
        }
        $delete->delete();
        $notification = [
            'message' => $delete->name . ' deleted successfully',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);
    }

    //============= Remittance actions
    public function remitPending(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $items = RemittanceOrder::where('status', 'pending')->latest()->get();
        return view('remittance.remittance_order', compact('dashboard', 'settings', 'items'));
    }

    public function remitSuccess(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $items = RemittanceOrder::where('status', 'accepted')->latest()->get();
        return view('remittance.remittance_order', compact('dashboard', 'settings', 'items'));
    }

    public function remitRejected(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $items = RemittanceOrder::where('status', 'cancelled')->latest()->get();
        return view('remittance.remittance_order', compact('dashboard', 'settings', 'items'));
    }

    public function remitApproved(Request $request): RedirectResponse
    {
        $notificationController = new FcmController();

        $update = RemittanceOrder::where('id', $request->success_id)->first();
        $update->update([
            'status' => 'accepted',
            'recipient_number' => $request->change_number ?? $update->recipient_number,
        ]);

        $notificationController->sendFcmNotification($update->user_id, 'Remittance ' . $update->bank_name, 'Your ' . $update->bank_name . ' service is approved');

        $notification = [
            'message' => 'This ' . $update->bank_name . ' service is approved successfully',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);

    }

    public function remitOrderRejected(Request $request): RedirectResponse
    {
        $notificationController = new FcmController();

        $validator = Validator::make($request->all(), [
            'reason' => 'required'
        ], [
            'reason.required' => 'Write Mobile Banking Rejected Reason',
        ]);

        if ($validator->fails()) {

            $notification = [
                'message' => $validator->errors()->first(),
                'alert-type' => 'error'
            ];

            return redirect()->back()->with($notification);
        }

        $update = RemittanceOrder::where('id', $request->reject_id)->first();
        $update->update([
            'status' => 'cancelled',
            'reason' => $request->reason
        ]);

        $refundAmount = $update->transfer_amount + $update->fee;
        $update->user->balance += $refundAmount;
        $update->user->update();

        $notificationController->sendFcmNotification($update->user_id, 'Remittance ' . $update->bank_name, 'Your ' . $update->bank_name . ' service is rejected & refunded successfully');

        $notification = [
            'message' => 'This ' . $update->bank_name . ' service is rejected successfully!',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);

    }

    public function remitDelete($id): RedirectResponse
    {
        $delete = RemittanceOrder::findOrFail($id);
        $delete->delete();

        $notification = [
            'message' => 'This service is deleted successfully!',
            'alert-type' => 'success'
        ];
        return redirect()->back()->with($notification);
    }
}
