<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Controllers\FcmController;
use App\Http\Requests\Auth\LoginRequest;
use App\Models\Settings;
use App\Models\User;
use App\Models\UserDocument;
use Carbon\Carbon;
use GeoIp2\Record\Country;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\View\View;

class UserController extends Controller
{
    public function index(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $users = User::all()->except(Auth::id())->sortByDesc('created_at');
        return view('users.index', compact('dashboard', 'settings', 'users'));
    }

    public function addUser(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        return view('users.add', compact('dashboard', 'settings'));
    }

    public function editUser($id): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);
        $user = User::findOrFail($id);
        return view('users.edit', compact('dashboard', 'settings', 'user'));
    }

    public function storeUser(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'pin' => 'required|numeric|digits:4',
            'password' => 'required|string|min:6',
            'number' => 'required|min:10|unique:users,phone',
            'email' => 'required|unique:users,email',
        ], [
            'name.required' => 'Name is required',
            'email.required' => 'Email is required',
            'email.unique' => 'Email already exists',
            'password.required' => 'Password is required',
            'number.required' => 'Phone is required',
            'number.unique' => 'Phone is already exists',
            'password.min' => 'Minimum password is 6 characters',
            'pin.required' => 'Pin is required',
            'pin.digits' => 'Pin must be 4 digits',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $user = User::create([
            'accountID' => Auth::user()->id,
            'name' => $request->name,
            'email' => $request->email,
            'username' => $this->crateUniqueUsername($request->name),
            'phone' => $request->number,
            'pin' => $request->pin,
            'password' => Hash::make($request->password),
        ]);

        $documentUpload = new UserDocument();
        $documentUpload->user_id = $user->id;
        $documentUpload->save();

        $notification = [
            'message' => $request->name . ' added successfully',
            'alert-type' => 'success',
        ];
        return back()->with($notification);
    }

    public function updateUser(Request $request, $id): RedirectResponse
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'pin' => 'required|string|digits:4',
            'email' => 'required|email|unique:users,email,' . $id,
            'number' => 'required|min:10|unique:users,phone,' . $id,
            'role' => 'nullable',
        ], [
            'name.required' => 'Name is required',
            'number.required' => 'Phone is required',
            'number.unique' => 'Phone is already exists',
            'nid_number.required' => 'NID or Passport number is required',
            'nid_number.unique' => 'This NID or Passport number is already exists',
            'role.required' => 'Role is required',
            'pin.required' => 'Pin is required',
            'pin.digits' => 'Pin must be 4 digits',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $user = User::findOrFail($id);
        $user->name = $request->name;
        $user->email = $request->email;
        $user->phone = $request->number;
        $user->pin = $request->pin;
        $user->status = $request->status == 'on' ? 'active' : 'inactive';
        $user->update();

        $notification = [
            'message' => $user->name . ' successfully updated',
            'alert-type' => 'success',
        ];
        return back()->with($notification);
    }

    public function updatePassword(Request $request, $id): RedirectResponse
    {
        $validator = Validator::make($request->all(), [
            'new_password' => 'required|string|min:6',
            'confirm_password' => 'required|string|min:6|same:new_password',
        ], [
            'new_password.required' => 'New Password is required',
            'new_password.min' => 'New Password must be at least 4 characters',
            'confirm_password.required' => 'Confirm Password is required',
            'confirm_password.min' => 'Confirm Password must be at least 4 characters',
            'confirm_password.same' => 'Confirm Password does not match',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator);
        }

        $updatePass = User::findOrFail($id);
        $updatePass->password = Hash::make($request->confirm_password);
        $updatePass->update();

        $notification = [
            'message' => $updatePass->name . "'s password has been changed",
            'alert-type' => 'success',
        ];
        return redirect()->back()->with($notification);
    }

    public function updateBalance(Request $request, $id): RedirectResponse
    {
        $validator = Validator::make($request->all(), [
            'balance' => 'required|numeric|min:50',
            's_balance' => 'required|in:debit,credit',
        ], [
            'balance.required' => 'Balance is required',
            'balance.numeric' => 'Balance must be a number',
            'balance.min' => 'Minimum balance is 50 BDT',
            's_balance.required' => 'Please select Debit or Credit Balance',
            's_balance.in' => 'Invalid balance type',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $user = User::findOrFail($id);
        $balanceChange = (float)$request->balance;

        if ($request->s_balance === 'debit') {
            if ($user->balance < $balanceChange) {
                $notification = [
                    'message' => 'Insufficient balance for debit',
                    'alert-type' => 'danger',
                ];
                return redirect()->back()->with($notification)->withInput();
            }
            $user->balance -= $balanceChange;
            $msg = "$balanceChange BDT has been deducted from {$user->name}'s account";

        } elseif ($request->s_balance === 'credit') {
            $user->balance += $balanceChange;
            $msg = "$balanceChange BDT added to {$user->name}'s account";
        }

        $user->update();

        $notification = [
            'message' => $msg,
            'alert-type' => 'success',
        ];
        return redirect()->back()->with($notification);
    }

    public function kycUpdate(Request $request, $id): RedirectResponse
    {
        $kyc = User::findOrFail($id);
        $kyc->document->status = $request->kyc_status;
        $kyc->document->save();

        $notificationController = new FcmController();
        $status = optional($kyc->document)->status ?? 'Unknown';

        if ($status === 'Approved') {
            $message = "Dear {$kyc->name}, congratulations! Your KYC has been successfully approved. You can now enjoy all features of our platform.";
        } elseif ($status === 'Rejected') {
            $message = "Dear {$kyc->name}, unfortunately, your KYC verification has been rejected. Please recheck your submitted documents and try again.";
        }

        $notificationController->sendFcmNotification($kyc->id, $kyc->name, $message);

        $notification = [
            'message' => $kyc->name . ' status has been updated',
            'alert-type' => 'success',
        ];

        return redirect()->back()->with($notification);
    }

    public function deleteUser($id): RedirectResponse
    {
        $delete = User::findOrFail($id);
        if ($delete->profile && File::exists(public_path($delete->profile))) {
            File::delete(public_path($delete->profile));
        }
        $delete->delete();
        $notification = [
            'message' => $delete->name . ' deleted successfully',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);
    }


    private function crateUniqueUsername($name): string
    {
        $username = Str::slug($name);
        $count = User::where('username', 'LIKE', "{username}%")->count();

        if ($count === 0) {
            return $username;
        }
        return $username . '-' . ($count + 1);
    }

}
