<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Company;
use App\Models\Config;
use App\Models\Package;
use App\Models\Permission;
use App\Models\PromotionBanner;
use App\Models\ResetPassword;
use App\Models\SmsSetting;
use App\Models\User;
use App\Models\UserDocument;
use App\Models\VerifyNumber;
use App\Models\VideoTutorial;
use Carbon\Carbon;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use PhpParser\Node\Scalar\String_;
use function Symfony\Component\String\u;

class AuthController extends Controller
{

    public function checkNumber(Request $request): JsonResponse
    {
        $user = User::where('phone', $request->phone)->first();

        if ($user) {
            return response()->json([
                'status' => false,
                'message' => 'This phone number is already registered.'
            ]);
        }

        VerifyNumber::where('number', $request->phone)->delete();

        $this->numberVerify($request->phone);
        return response()->json([
            'status' => true,
            'message' => 'Phone number is available. Proceed to OTP verification.'
        ]);
    }

    public function numberVerify(string $number): string
    {
        $formatted = Carbon::now()->subMinutes(5)->toDateTimeString();
        VerifyNumber::where('created_at', '<=', $formatted)->delete();

        $otp = sprintf('%06d', rand(0, 999999)); // Generate a 6-digit OTP
        VerifyNumber::create([
            'number' => $number,
            'otp' => $otp,
            'created_at' => Carbon::now()
        ]);

        Log::info("OTP for $number is $otp");

        return $otp;
    }

    public function getOtp(Request $request): JsonResponse
    {
        $value = $request->header('code-key');
        if (!$value) {
            return response()->json([
                'status' => false,
                'message' => 'Invalid request.']);
        }

        $get_otp = VerifyNumber::where('number', $value)->first();
        $data = [
            'otp' => $get_otp->otp,
        ];

        Log::info($data);

        if ($get_otp) {
            return response()->json([
                'status' => true,
                'data' => $data
            ]);
        } else {
            return response()->json([
                'status' => false,
                'message' => 'Unauthorized access request.'
            ]);
        }

    }

    public function verifyPhoneNumber(Request $request): JsonResponse
    {
        $formatted = Carbon::now()->subMinutes(5)->toDateTimeString();
        VerifyNumber::where('created_at', '<=', $formatted)->delete();
        $numberVerification = VerifyNumber::where('number', $request->phone)->where('otp', $request->otp)->first();
        if (!$numberVerification) {
            return response()->json([
                'status' => false,
                'data' => [
                    'error' => 'Invalid OTP Or Expired',
                ]
            ]);
        } else {
            ResetPassword::where('number', $request->phone)->delete();
            return response()->json([
                'status' => true,
                'data' => [
                    'success' => 'Your OTP is Valid',
                ]
            ]);
        }
    }


    public function register(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required',
            'email' => 'required|email|unique:users',
            'phone' => 'required|unique:users',
            'pin_code' => 'required|digits:4',
            'password' => 'required|min:6',
            'document_type' => 'required|in:Nid Card Document,Driving Document,Birth Certificate Document',
            'document_front' => 'required|string',
            'document_back' => in_array($request->input('document_type'), ['Nid Card Document', 'Driving Document']) ? 'required' : 'nullable',
        ], [
            'name.required' => 'Please enter your name.',
            'email.required' => 'Please enter your email.',
            'email.email' => 'Please enter a valid email.',
            'email.unique' => 'This email is already registered.',
            'phone.required' => 'Please enter your phone number.',
            'phone.unique' => 'This phone number is already registered.',
            'pin_code.required' => 'Please enter your pin code.',
            'pin_code.digits' => 'Pin code must be 4 characters.',
            'password.required' => 'Please enter your password.',
            'password.min' => 'Password minimum 6 characters.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'errors' => $validator->errors()
            ]);
        }

        // ইউজারনেম জেনারেশন
        $usernameBase = Str::lower(str_replace(' ', '', $request->name));
        $username = $usernameBase;
        $counter = 1;
        while (User::where('username', $username)->exists()) {
            $username = $usernameBase . $counter;
            $counter++;
        }

        $user = User::create([
            'name' => $request->name,
            'username' => $username,
            'email' => $request->email,
            'phone' => $request->phone,
            'pin' => $request->pin_code,
            'password' => Hash::make($request->password),
        ]);

        if (!$user) {
            return response()->json([
                'status' => false,
                'message' => 'Registration Failed. Please try again.'
            ]);
        }

        $documentUpload = new UserDocument();
        $documentUpload->user_id = $user->id;
        $documentUpload->type = $request->document_type;

        $frontDecoded = base64_decode($request->document_front, true);
        $fileNameFront = $request->phone . '_front_' . time() . '.png';
        $pathFront = 'upload/documents/' . $fileNameFront;
        file_put_contents(public_path($pathFront), $frontDecoded);
        $documentUpload->document_one = $pathFront;

        if ($request->document_back) {
            $backDecoded = base64_decode($request->document_back, true);
            $fileNameBack = $request->phone . '_back_' . time() . '.png';
            $pathBack = 'upload/documents/' . $fileNameBack;
            file_put_contents(public_path($pathBack), $backDecoded);
            $documentUpload->document_two = $pathBack;
        }
        $documentUpload->save();

        return response()->json([
            'status' => true,
            'message' => 'Registration Successful.'
        ]);
    }


    // User Login
    public function login(Request $request): JsonResponse
    {
        $request->validate([
            'phone' => 'required|string',
            'password' => 'required|string',
        ]);

        $user = User::where('phone', $request->phone)->whereNotIn('role', ['admin'])->first();

        if (!$user) {
            return response()->json([
                'status' => false,
                'message' => ['phone' => 'Incorrect phone number']
            ]);
        }

        if (Hash::check($request->password, $user->password)) {
            $user->tokens()->delete();
            $token = $user->createToken('api', ['*'], now()->addDays(10))->plainTextToken;
            $user->logged_in_location = $request->input('location');
            $user->save();
            return response()->json([
                'status' => true,
                'data' => ['token' => $token]
            ]);
        } else {
            return response()->json([
                'status' => false,
                'message' => ['password' => 'Your password is wrong']
            ]);
        }
    }

    public
    function accountInfo(): JsonResponse
    {
        $promotion = PromotionBanner::inRandomOrder()->first();
        $data = [
            'id' => Auth::user()->id,
            'name' => Auth::user()->name,
            'phone' => Auth::user()->phone,
            'banner' => $promotion ? $promotion->banner : null
        ];

        return response()->json([
            'status' => true,
            'data' => $data
        ]);
    }

    public
    function pin(Request $request): JsonResponse
    {
        $user = auth()->user();
        if ($request->get('pin') == $user->pin) {
            return response()->json([
                'status' => true,
                'data' => 'Success'

            ]);
        } else {
            return response()->json([
                'status' => false,
                'error' => 'Pin is wrong',
            ]);
        }
    }


    public
    function logout(): JsonResponse
    {
        if ($user = Auth::user()) {
            $user->currentAccessToken()->delete();
            return response()->json([
                'status' => true,
                'message' => 'Logout successful',
            ]);
        }
        return response()->json([
            'status' => false,
            'message' => 'No authenticated user found',
        ]);
    }


// For user created according his type account only
    public
    function createAccount(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required',
            'email' => 'required|email|unique:users',
            'phone' => 'required|unique:users',
            'nid_passport' => 'required|unique:users',
            'country' => 'required',
            'pin_code' => 'required|min:4|max:4',
            'password' => 'required|min:6',
        ], [
            'name.required' => 'Please enter your name.',
            'email.required' => 'Please enter your email.',
            'email.email' => 'Please enter a valid email.',
            'email.unique' => 'This email is already registered.',
            'phone.required' => 'Please enter your phone number.',
            'phone.unique' => 'This phone number is already registered.',
            'nid_passport.require' => 'Please enter a NID or Password Number.',
            'nid_passport.unique' => 'This NID or Password Number is already registered.',
            'country.required' => 'Please enter your country.',
            'pin_code.required' => 'Please enter your pin code.',
            'pin_code.min' => 'Pin code must be 4 characters.',
            'pin_code.max' => 'Pin code must be 4 characters.',
            'password.required' => 'Please enter your password.',
            'password.min' => 'Password minimum 6 characters.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'errors' => $validator->errors()
            ]);
        }

        $user = Auth::user();
        $requestedLevel = $request->get('level');

        if (
            ($user->role === 'dgm' && $requestedLevel !== 'dgm') ||
            ($user->role === 'retailer' && $requestedLevel !== 'retailer') ||
            ($user->role === 'subAdmin' && !in_array($requestedLevel, ['dgm', 'retailer']))
        ) {
            return response()->json([
                'status' => false,
                'level_error' => 'You are not authorized to create this account type.'
            ]);
        }

        $usernameBase = Str::lower(str_replace(' ', '', $request->name));
        $username = $usernameBase;

        $counter = 1;
        while (User::where('username', $username)->exists()) {
            $username = $usernameBase . $counter;
            $counter++;
        }

        User::create([
            'accountID' => $user->id,
            'name' => $request->name,
            'username' => $username,
            'email' => $request->email,
            'phone' => $request->phone,
            'location' => $request->country,
            'nid_passport' => $request->nid_passport,
            'pin' => $request->pin_code,
            'password' => Hash::make($request->password),
            'role' => $requestedLevel,
        ]);

        return response()->json([
            'status' => true,
            'message' => 'Account created successfully.',
        ]);
    }

    public
    function getAccounts(): JsonResponse
    {
        $accounts = User::where('accountID', Auth::id())->get();

        $data = $accounts->map(function ($account) {
            return [
                'id' => $account->id,
                'name' => $account->name,
                'email' => $account->email,
                'phone' => $account->phone,
                'profile' => $account->profile,
                'country' => $account->location
            ];
        });

        return response()->json([
            'status' => true,
            'data' => $data
        ]);
    }

//for config
    public
    function config(Request $request): JsonResponse
    {
        $value = $request->header('code-key');

        if (!$value) {
            return response()->json([
                'status' => false,
                'message' => 'Missing required headers.'
            ]);
        }

        $config = Config::first();
        $permissions = Permission::first();

        if ($value !== env('APP_PACKAGE_NAME')) {
            return response()->json([
                'status' => false,
                'message' => 'Unauthorized header values.'
            ]);
        }

        $data = [
            'name' => $config->packageName,
            'secret_key' => $config->secret_key,
            'app_secret' => $config->app_secret,
            'location' => $permissions->status
        ];

        return response()->json([
            'status' => true,
            'data' => $data
        ]);
    }

//send OTP
    /* public function sendOtp(Request $request): JsonResponse
     {
         $formatted = Carbon::now()->subMinutes(5)->toDateTimeString();
         ResetPassword::where('created_at', '<=', $formatted)->delete();

         $existingOTP = ResetPassword::where('phone', $request->phone)->first();
         if ($existingOTP) {
             return response()->json([
                 'status' => false,
                 'data' => [
                     'error' => 'An OTP has already been sent to this Mobile Number. Please check your message.',
                 ]
             ]);
         } else {
             $user = User::where('phone', $request->phone)->first();
             $otp = sprintf('%04d', rand(0, 9999));

             if (!$user) {
                 return response()->json([
                     'status' => false,
                     'message' => 'failed',
                     'data' => [
                         'error' => 'This Mobile number could not be found, please try again.',
                     ]
                 ]);
             }

             ResetPassword::create([
                 'number' => $request->phone,
                 'otp' => $otp,
                 'created_at' => Carbon::now()
             ]);

             return response()->json([
                 'status' => true,
                 'data' => [
                     'success' => 'OTP has been sent to your Mobile Number. Check your Inbox.'
                 ]
             ]);
         }

     }*/

    public
    function sendOtp(Request $request): JsonResponse
    {
        $formatted = Carbon::now()->subMinutes(5)->toDateTimeString(); // Expire old OTPs after 5 minutes
        ResetPassword::where('created_at', '<=', $formatted)->delete();

        $existingOTP = ResetPassword::where('number', $request->phone)->first();
        if ($existingOTP) {
            return response()->json([
                'status' => false,
                'already_used' => true,
                'data' => [
                    'error' => 'An OTP has already been sent to this mobile number. Please check your messages or wait until it expires.',
                ]
            ]);
        } else {
            $user = User::where('phone', $request->phone)->first();
            $otp = sprintf('%04d', rand(0, 9999)); // Generate a 4-digit OTP

            if (!$user) {
                return response()->json([
                    'status' => false,
                    'data' => [
                        'error' => 'This mobile number was not found. Please verify and try again.',
                    ]
                ]);
            }

            $service = SmsSetting::findOrFail(1);
            if ($service->api_key == null && $service->sender_id == null) {
                return response()->json([
                    'status' => false,
                    'data' => [
                        'error' => 'Configuration Problem, Contact to admin',
                    ]
                ]);
            }

            ResetPassword::create([
                'number' => $request->phone,
                'otp' => $otp,
                'created_at' => Carbon::now()
            ]);


            $api_key = $service->api_key;
            $sender_id = $service->sender_id;
            $message = "Dear, " . $user->name . "\nYour OTP is $otp. Please use this code within 5 minutes to complete your verification.";

            $response = Http::get("http://bulksmsbd.net/api/smsapi", [
                'api_key' => $api_key,
                'senderid' => $sender_id,
                'number' => $request->phone,
                'message' => $message,
                'type' => 'text'
            ]);

            if ($response->successful()) {
                return response()->json([
                    'status' => true,
                    'data' => [
                        'success' => 'An OTP has been sent to your mobile number. It will expire in 5 minutes.',
                    ]
                ]);
            } else {
                return response()->json([
                    'status' => false,
                    'data' => [
                        'error' => 'Failed to send OTP. Please try again later.',
                    ]
                ]);
            }
        }
    }

    public
    function verifyOtp(Request $request): JsonResponse
    {
        $formatted = Carbon::now()->subMinutes(5)->toDateTimeString();
        ResetPassword::where('created_at', '<=', $formatted)->delete();

        $otpValidation = ResetPassword::where('number', $request->phone)->where('otp', $request->otp)->first();

        if (!$otpValidation) {
            return response()->json([
                'status' => false,
                'data' => [
                    'error' => 'Invalid OTP Or Expired',
                ]
            ]);
        } else {
            ResetPassword::where('number', $request->phone)->delete();
            return response()->json([
                'status' => true,
                'data' => [
                    'success' => 'Your OTP is Valid',
                ]
            ]);
        }
    }

    public
    function changePassword(Request $request): JsonResponse
    {
        if ($request->get('service_code') == 6484) {
            $user = User::where('phone', $request->phone)->first();
            $user->password = Hash::make($request->get('password'));
            $user->update();
            return response()->json([
                'status' => true,
                'message' => 'Your new password has been updated',
            ]);
        } else {
            return response()->json([
                'status' => false,
                'errors' => 'Unauthorized access',
            ]);
        }
    }


    public
    function token(Request $request): JsonResponse
    {
        $request->validate([
            'fcm_token' => 'required|string',
        ]);

        $user = User::findOrFail(Auth::id());
        $user->fcm_token = $request->get('fcm_token');
        $user->save();

        return response()->json([
            'status' => true,
            'message' => 'Token updated'
        ]);
    }

    public
    function deleteDrivePackage()
    {
        $delete = Package::findOrFail(3);
        $delete->delete();
        $data = [
            'Package' => $delete,
            'id' => 3
        ];
        Log::info($data);

        return response()->json([
            'status' => true,
            'message' => 'Deleted successfully',
        ]);
    }

    public
    function video(): JsonResponse
    {
        $show = VideoTutorial::first();
        $data = [
            'reg_video' => $show->reg_video
        ];

        return response()->json([
            'status' => true,
            'data' => $data
        ]);
    }


}
