<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Livewire\Forms\LoginForm;
use App\Models\AddMoneyOrder;
use App\Models\Bank;
use App\Models\BankTransfer;
use App\Models\Billing;
use App\Models\BillPayment;
use App\Models\Company;
use App\Models\DriveOrder;
use App\Models\ExchangeRates;
use App\Models\Extra;
use App\Models\GlobalRecharge;
use App\Models\GlobalRechargeOrder;
use App\Models\MBankingOrder;
use App\Models\MobileBanking;
use App\Models\NoticeMarquee;
use App\Models\Offer;
use App\Models\PromotionBanner;
use App\Models\Recharge;
use App\Models\Remittance;
use App\Models\RemittanceOrder;
use App\Models\SendmoneyOrder;
use App\Models\Settings;
use App\Models\Support;
use App\Models\User;
use App\Models\UserDocument;
use App\Models\Videos;
use App\Models\VideoTutorial;
use Carbon\Carbon;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use PHPUnit\Framework\TestStatus\Notice;
use function Symfony\Component\String\s;

class DashboardController extends Controller
{
    public function infos(): JsonResponse
    {
        $info = User::findOrFail(Auth::id());
        $data = [
            'profile' => url($info->profile),
            'name' => $info->name,
            'username' => $info->username,
            'phone' => $info->phone,
            'balance' => $info->balance,
        ];

        $promotions = PromotionBanner::all()->sortByDesc('created_at');
        $p_data = $promotions->map(function ($promotions) {
            return [
                'banner' => url($promotions->banner),
            ];
        });

        $notice = NoticeMarquee::findOrFail(1);
        $n_data = [
            'notice' => $notice->notice_text,
        ];

        $mBanking = MobileBanking::all();
        $m_data = $mBanking->map(function ($mBanking) {
            return [
                'name' => $mBanking->name,
                'logo' => url($mBanking->logo),
            ];
        });

        $offers = Offer::all();
        $offer_data = $offers->map(function ($offer) {
            return [
                'id' => $offer->id,
                'title' => $offer->title,
                'url' => $offer->url,
                'logo' => url($offer->image)
            ];
        });


        $settings = Settings::findOrFail(1);
        $s_data = [
            'terms' => $settings->terms_url,
            'privacy' => $settings->privacy_url,
            'noway' => 'https://codecollection.net/',
        ];

        $dashboard_notice = \App\Models\Notice::findOrFail(1);
        $d_notice = [
            'notice' => $dashboard_notice->notice,
            'package_notice' => $dashboard_notice->package_notice,
            'rate_notice' => $dashboard_notice->rate_notice,
        ];

        $videoTutorial = VideoTutorial::first();
        $videoData = [
            'recharge_video' => $videoTutorial->recharge_video,
            'send_money_video' => $videoTutorial->send_money_video,
            'pay_bill_video' => $videoTutorial->pay_bill_video,
            'add_money_video' => $videoTutorial->add_money_video,
            'bank_transfer_video' => $videoTutorial->bank_transfer_video,
            'mobile_banking_video' => $videoTutorial->mobile_banking_video,
            'remittance_video' => $videoTutorial->remittance_video,
        ];

        $verify = UserDocument::where('user_id', auth()->id())->first();

        return response()->json(
            [
                'status' => true,
                'verify' => $verify->status,
                'data' => $data,
                'banners' => $p_data,
                'notice' => $n_data,
                'm_banking' => $m_data,
                'settings' => $s_data,
                'dashboard_notice' => $d_notice,
                'offers' => $offer_data,
                'videos' => $videoData,
            ]
        );
    }

    public function profile(): JsonResponse
    {
        $info = User::findOrFail(Auth::id());
        $data = [
            'profile' => $info->profile,
            'name' => $info->name,
            'email' => $info->email,
            'phone' => $info->phone,
        ];
        return response()->json(
            [
                'status' => true,
                'data' => $data
            ]
        );
    }

    public function updateProfile(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'phone' => 'nullable|unique:users,phone,' . Auth::id(),
        ], [
            'phone.required' => 'Phone is required',
            'phone.unique' => 'Phone already exists',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'errors' => $validator->errors(),
            ]);
        }

        $user = User::findOrFail(Auth::id());
        $user->name = $request->get('name');
        $user->email = $request->get('email');
        $user->phone = $request->get('phone');

        if ($request->has('profile')) {
            $updateImage = base64_decode($request->profile);
            $fileName = $user->phone . "_" . time() . '.png';
            $path = 'upload/users/' . $fileName;
            file_put_contents(public_path($path), $updateImage);

            if ($user->profile && File::exists(public_path($user->profile))) {
                File::delete(public_path($user->profile));
            }
            $user->profile = $path;
        }

        $user->update();
        return response()->json([
            'status' => true,
            'message' => 'Your profile has been updated',
        ]);
    }

    public function updatePassword(Request $request): JsonResponse
    {
        $user = auth()->user();
        if (!Hash::check($request->get('oldPassword'), $user->password)) {
            return response()->json([
                'success' => false,
                'errors' => 'Your old pin is incorrect',
            ]);
        }

        $user->password = Hash::make($request->get('newPassword'));
        $user->update();
        return response()->json([
            'success' => true,
            'message' => 'Your new pin has been updated',
        ]);
    }

    public function updatePin(Request $request): JsonResponse
    {
        $user = auth()->user();
        if ($request->get('oldPin') != $user->pin) {
            return response()->json([
                'success' => false,
                'errors' => 'Your old pin is incorrect',
            ]);
        }

        $user->pin = $request->newPin;
        $user->update();
        return response()->json([
            'success' => true,
            'message' => 'Your new pin has been updated',
        ]);
    }

    public function allBank(): JsonResponse
    {
        $banks = Bank::all()->sortByDesc('created_at');
        $data = $banks->map(function ($bank) {
            return [
                'name' => $bank->name,
            ];
        });

        return response()->json([
            'status' => true,
            'data' => $data,
        ]);
    }

    public function remittanceBank(): JsonResponse
    {
        $banks = Remittance::all()->sortByDesc('created_at');
        $data = $banks->map(function ($bank) {
            return [
                'name' => $bank->name,
            ];
        });

        return response()->json([
            'status' => true,
            'data' => $data,
        ]);
    }

    public function globalOperator(): JsonResponse
    {
        $recharges = GlobalRecharge::all()->sortByDesc('created_at');
        $data = $recharges->map(function ($recharge) {
            return [
                'name' => $recharge->name,
            ];
        });

        return response()->json([
            'status' => true,
            'data' => $data,
        ]);
    }

    public function billProvider(): JsonResponse
    {
        $billProviders = Billing::all()->sortByDesc('created_at');
        $data = $billProviders->map(function ($billProvider) {
            return [
                'name' => $billProvider->name,
            ];
        });

        return response()->json([
            'status' => true,
            'data' => $data,
        ]);
    }


    public function history(): JsonResponse
    {
        $userId = Auth::id();

        // Retrieve and format each type of history data
        $addMoneyHistory = AddMoneyOrder::where('user_id', $userId)->get()->map(function ($item) {
            return [
                'id' => $item->id,
                'history_type' => 'Add Money',
                'amount' => $item->amount,
                'status' => $item->status,
                'details' => array_merge([
                    'Method Name' => $item->method_name,
                    'Holder Name' => $item->holder_name,
                    'Account Number' => $item->account_number,
                    'Method' => $item->add_money,
                ], $item->reason ? ['Reason' => $item->reason] : []),
                'created_at' => $item->created_at
            ];
        });

        $rechargeHistory = Recharge::where('user_id', $userId) // Only include if reason is not null
        ->get()->map(function ($item) {
            return [
                'id' => $item->id,
                'history_type' => 'Recharge',
                'amount' => $item->amount,
                'status' => $item->status,
                'details' => array_merge([
                    'Operator' => $item->operator_name,
                    'Number' => $item->number,
                ], $item->reason ? ['Reason' => $item->reason] : [], $item->commission ? ['Commission' => $item->commission . ' BDT'] : []),
                'created_at' => $item->created_at
            ];
        });

        $remittanceHistory = RemittanceOrder::where('user_id', $userId)->get()->map(function ($item) {
            return [
                'id' => $item->id,
                'history_type' => 'Remittance',
                'amount' => $item->transfer_amount,
                'status' => $item->status,
                'details' => array_merge([
                    'Recipient Name' => $item->recipient_name,
                    'Bank Name' => $item->bank_name,
                ], $item->reason ? ['Reason' => $item->reason] : [], $item->fee ? ['Fee' => $item->fee . ' BDT'] : []),
                'created_at' => $item->created_at
            ];
        });

        $sendMoneyHistory = SendmoneyOrder::where('user_id', $userId)->get()->map(function ($item) {
            return [
                'id' => $item->id,
                'history_type' => 'Send Money',
                'amount' => $item->amount,
                'status' => $item->status,
                'details' => array_merge([
                    'Recipient Number' => $item->ac_number,
                ], $item->reason ? ['Reason' => $item->reason] : []),
                'created_at' => $item->created_at
            ];
        });

        $billPaymentHistory = BillPayment::where('user_id', $userId)->get()->map(function ($item) {
            return [
                'id' => $item->id,
                'history_type' => 'Bill Payment',
                'amount' => $item->amount,
                'status' => $item->status,
                'details' => array_merge([
                    'Bill' => $item->billing_name,
                    'Account Number' => $item->account_number,
                ], $item->reason ? ['Reason' => $item->reason] : [], $item->commission ? ['Commission' => $item->commission . ' BDT'] : [], $item->fee ? ['Fee' => $item->fee . ' BDT'] : []),
                'created_at' => $item->created_at
            ];
        });

        $bankTransferHistory = BankTransfer::where('user_id', $userId)->get()->map(function ($item) {
            return [
                'id' => $item->id,
                'history_type' => 'Bank Transfer',
                'amount' => $item->amount,
                'status' => $item->status,
                'details' => array_merge([
                    'Bank Name' => $item->bank_name,
                    'Account Name' => $item->ac_name,
                ], $item->reason ? ['Reason' => $item->reason] : [], $item->commission ? ['Commission' => $item->commission . ' BDT'] : [], $item->fee ? ['Fee' => $item->fee . ' BDT'] : []),
                'created_at' => $item->created_at
            ];
        });

        $mBankingHistory = MBankingOrder::where('user_id', $userId)->get()->map(function ($item) {
            return [
                'id' => $item->id,
                'history_type' => 'Mobile Banking',
                'amount' => $item->amount,
                'status' => $item->status,
                'details' => array_merge([
                    'MBanking Name' => $item->mbank_name,
                    'Number' => $item->mbank_number,
                ], $item->reason ? ['Reason' => $item->reason] : [], $item->last_number ? ['Last Number' => $item->last_number] : [], $item->trx_id ? ['Trx ID' => $item->trx_id] : [], $item->commission ? ['Commission' => $item->commission . ' BDT'] : [], $item->fee ? ['Fee' => $item->fee . ' BDT'] : []),
                'created_at' => $item->created_at
            ];
        });

        // Combine and sort all histories
        $allHistory = collect()
            ->merge($addMoneyHistory)
            ->merge($rechargeHistory)
            ->merge($remittanceHistory)
            ->merge($sendMoneyHistory)
            ->merge($billPaymentHistory)
            ->merge($bankTransferHistory)
            ->merge($mBankingHistory)
            ->sortByDesc('created_at')
            ->map(function ($item) {
                // Format the created_at field after sorting
                $item['created_at'] = Carbon::parse($item['created_at'])->format('D, d M Y h:i:s a');
                return $item;
            })
            ->values();

        return response()->json([
            'status' => true,
            'data' => $allHistory
        ]);
    }

    public function support(): JsonResponse
    {
        $support = Support::findOrFail(1);
        $data = [
            'messenger' => $support->messenger,
            'telegram' => $support->telegram,
            'whatsapp' => $support->whatsapp,
            'youtube' => $support->youtube,
            'about' => $support->about,
        ];

        return response()->json([
            'status' => true,
            'data' => $data
        ]);
    }

    public function company(): JsonResponse
    {
        $company = Company::first();
        $data = [
            'trade_license' => env('APP_URL') . '/' . $company->trade_license,
            'certificate' => env('APP_URL') . '/' . $company->certificate,
            'about_company' => $company->about_company,
        ];

        return response()->json([
            'status' => true,
            'data' => $data
        ]);
    }

    public function extras(): JsonResponse
    {
        $extra = Extra::findOrFail(1);
        $data = [
            'ecommerce' => $extra->web_address,
            'youtube' => $extra->tutorial,
        ];

        return response()->json([
            'status' => true,
            'data' => $data
        ]);
    }

    public function admins(): JsonResponse
    {
        $admins = User::where('role', 'admin')->get();

        $data = $admins->map(function ($admin) {
            return [
                'id' => $admin->id,
                'name' => $admin->name,
                'email' => $admin->email,
            ];
        });

        return response()->json([
            'status' => true,
            'data' => $data
        ]);

    }

}
