<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Controllers\FcmController;
use App\Models\ApiRecharge;
use App\Models\BankTransfer;
use App\Models\BillPayment;
use App\Models\Commission;
use App\Models\DriveOrder;
use App\Models\GlobalRechargeOrder;
use App\Models\MBankingOrder;
use App\Models\Package;
use App\Models\Recharge;
use App\Models\RemittanceOrder;
use App\Models\SendmoneyOrder;
use App\Models\User;
use GuzzleHttp\Client;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;

class OperatorController extends Controller
{
    public function packages(): JsonResponse
{
    // শুধুমাত্র status = active প্যাকেজ আনবে
    $offers = Package::where('status', 'active')->get();

    // যদি কোনো active প্যাকেজ না থাকে
    if ($offers->isEmpty()) {
        return response()->json([
            'status' => false,
            'message' => 'No active packages available',
            'data' => []
        ]);
    }

    $data = $offers->map(function ($item) {
        return [
            'id' => $item->id,
            'name' => $item->name,
            'discount' => $item->discount_price,
            'price' => $item->selling_price,
            'operator' => $item->operator,
            'type' => $item->package,
            'validity' => $item->validity,
        ];
    });

    return response()->json([
        'status' => true,
        'data' => $data
    ]);
}


    public function purchaseDrive(Request $request): JsonResponse
    {
        $user = auth()->user();
        $package = Package::findOrFail($request->get('package_id'));

        $commissionRate = Commission::findOrFail(3);
        $calculate = 0;
        if ($commissionRate->status == 'active') {
            $calculate = $request->get('amount') * ($commissionRate->amount / 100);
            $finalAmount = $request->get('amount') - $calculate;

        } else {
            $finalAmount = $request->get('amount');
        }

        if ($user->balance < $finalAmount) {
            return response()->json([
                'status' => false,
                'errors' => 'Insufficient balance'
            ]);
        }

        DriveOrder::create([
            'user_id' => $user->id,
            'package_id' => $package->id,
            'price' => $package->selling_price,
            'discount' => $package->discount_price,
            'number' => $request->get('number'),
            'pay' => $finalAmount,
            'type' => $package->package,
            'commission' => $calculate,
        ]);

        $user->balance -= $finalAmount;
        $user->update();

        $notification = new FcmController();
        $notification->sendAdminNotification("Drive Purchase", $package->name . ' is purchased from ' . $user->name);

        return response()->json([
            'status' => true,
            'message' => 'Purchase successful',
        ]);
    }

    public function remittance(Request $request): JsonResponse
    {
        $amount = $request->get('transferAmount');
        $fee = Commission::findOrFail(11);  // Fetch default or specific commission rate
        $calculate = 0;
        if ($fee->status == 'active') {
            $calculate = $amount * ($fee->amount / 100);
            $finalAmount = $amount + $calculate;
        } else {
            $finalAmount = $amount;
        }

        $user = Auth::user();
        if ($request->pin !== $user->pin) {
            return response()->json([
                'status' => false,
                'errors' => 'Your pin is wrong'
            ]);
        }

        if ($user->balance < $finalAmount) {
            return response()->json([
                'status' => false,
                'errors' => 'Insufficient balance'
            ]);
        }

        RemittanceOrder::create([
            'user_id' => $user->id,
            'recipient_number' => $request->get('recipientNumber'),
            'recipient_name' => $request->get('recipientName'),
            'bank_name' => $request->get('bankName'),
            'transfer_amount' => $request->get('transferAmount'),
            'sender_number' => $request->get('senderNumber'),
            'sender_name' => $request->get('senderName'),
            'fee' => $calculate,
        ]);

        $user->balance -= $finalAmount;
        $user->update();

        $notification = new FcmController();
        $notification->sendAdminNotification("Remittance request", $request->get('bankName') . ' is purchased from ' . $user->name);

        return response()->json([
            'status' => true,
            'message' => 'Remittance send successful',
        ]);
    }

    public function sendMoney(Request $request): JsonResponse
    {
        $notification = new FcmController();
        $user = Auth::user();
        $recipient = User::where('phone', $request->get('ac_number'))->first();

        if (!$recipient) {
            return response()->json([
                'status' => false,
                'errors' => 'User not found',
            ]);
        }

        if ($request->pin !== $user->pin) {
            return response()->json([
                'status' => false,
                'errors' => 'Your pin is wrong'
            ]);
        }

        if ($user->balance < $request->get('amount')) {
            return response()->json([
                'status' => false,
                'errors' => 'Insufficient balance',
            ]);
        }

        if ($recipient->phone === $user->phone) {
            return response()->json([
                'status' => false,
                'errors' => 'You cannot send money to your own account',
            ]);
        }

        DB::beginTransaction();

        try {
            $user->decrement('balance', $request->get('amount'));

            if ($recipient) {
                $recipient->increment('balance', $request->get('amount'));

                $notification->sendFcmNotification(
                    $recipient->id,
                    'Received Money',
                    'You received ' . $request->get('amount') . ' from ' . $user->phone
                );
            }

            SendmoneyOrder::create([
                'user_id' => $user->id,
                'ac_number' => $request->get('ac_number'),
                'amount' => $request->get('amount'),
                'status' => $recipient ? 'accepted' : 'pending'
            ]);

            $notification->sendAdminNotification("SendMoney", 'Send Money from ' . $user->first_name . ' ' . $user->last_name);

            DB::commit();

            return response()->json([
                'status' => true,
                'message' => 'Money sent successfully',
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'status' => false,
                'errors' => 'Transaction failed, please try again',
            ]);
        }
    }


    public function globalRechargeRequest(Request $request): JsonResponse
    {

        $commissionRate = Commission::findOrFail(2);  // Fetch default or specific commission rate
        $calculate = 0;
        if ($commissionRate->status == 'active') {
            $calculate = $request->get('amount') * ($commissionRate->amount / 100);
            $finalAmount = $request->get('amount') - $calculate;
        } else {
            $finalAmount = $request->get('amount');
        }

        $user = auth()->user();
        if ($user->balance < $finalAmount) {
            return response()->json([
                'status' => false,
                'errors' => 'Insufficient balance'
            ]);
        }

        GlobalRechargeOrder::create([
            'user_id' => $user->id,
            'number' => $request->get('number'),
            'operator' => $request->get('operator'),
            'amount' => $finalAmount,
            'commission' => $calculate,
        ]);

        $user->balance -= $finalAmount;
        $user->update();

        $notification = new FcmController();
        $notification->sendAdminNotification("Global recharge request", $request->get('operator') . ' request from ' . $user->name);

        return response()->json([
            'status' => true,
            'message' => 'Global recharge request send successful',
        ]);
    }


    public function mbankingRequest(Request $request): JsonResponse
    {
        $amount = $request->get('amount');
        $finalAmount = $amount;
        $chargeAmount = 0;
        $commissionAmount = 0;

        if ($request->type === 'Agent Number') {
            $cashoutCharge = Commission::findOrFail(10);
            $commissionRate = Commission::findOrFail(4);

            if ($cashoutCharge->status === 'active') {
                $chargeAmount = $amount * ($cashoutCharge->amount / 100);
                $finalAmount += $chargeAmount;  // Add charge to the final amount
            }

            if ($commissionRate->status === 'active') {
                $commissionAmount = $amount * ($commissionRate->amount / 100);
                $finalAmount -= $commissionAmount;
            }

        } elseif ($request->type === 'Personal Number') {
            $sendMoneyCharge = Commission::findOrFail(8);
            $commissionRate = Commission::findOrFail(4);

            if ($sendMoneyCharge->status === 'active') {
                $chargeAmount = $sendMoneyCharge->amount;
                $finalAmount += $chargeAmount;
            }

            if ($commissionRate->status === 'active') {
                $commissionAmount = $amount * ($commissionRate->amount / 100);
                $finalAmount -= $commissionAmount;
            }
        }

        $user = Auth::user();
        if ($user->pin != $request->pin) {
            return response()->json([
                'status' => false,
                'errors' => 'Your pin is wrong'
            ]);
        }

        if ($user->balance < $finalAmount) {
            return response()->json([
                'status' => false,
                'errors' => 'Insufficient balance'
            ]);
        }

        // Create the mobile banking order with calculated values
        MBankingOrder::create([
            'user_id' => $user->id,
            'mbank_name' => $request->get('mbank_name'),
            'mbank_number' => $request->get('number'),
            'amount' => $amount, // Original amount
            'transaction_type' => $request->get('type'),
            'commission' => $commissionAmount,
            'fee' => $chargeAmount,
        ]);

        // Deduct the final amount from the user's balance
        $user->balance -= $finalAmount;
        $user->update();

        // Send notification to admin about the mobile banking request
        $notification = new FcmController();
        $notification->sendAdminNotification(
            "Mobile Banking request",
            $request->get('mbank_name') . ' request from ' . $user->name
        );

        return response()->json([
            'status' => true,
            'message' => $request->get('mbank_name') . ' transfer request sent successfully',
        ]);
    }

    public function bankTransfer(Request $request): JsonResponse
    {
        $amount = $request->get('amount');
        $finalAmount = $amount;

        $charge = Commission::findOrFail(7);
        $commissionRate = Commission::findOrFail(6);

        $chargeAmount = 0;
        $commissionAmount = 0;

        if ($charge->status === 'active') {
            $chargeAmount = $amount * ($charge->amount / 100);
            $finalAmount += $chargeAmount;
        }

        if ($commissionRate->status === 'active') {
            $commissionAmount = $amount * ($commissionRate->amount / 100);
            $finalAmount -= $commissionAmount;
        }

        $user = Auth::user();
        if ($request->pin !== $user->pin) {
            return response()->json([
                'status' => false,
                'errors' => 'Your pin is wrong'
            ]);
        }

        if ($user->balance < $finalAmount) {
            return response()->json([
                'status' => false,
                'errors' => 'Insufficient balance'
            ]);
        }

        BankTransfer::create([
            'user_id' => $user->id,
            'bank_name' => $request->get('bank_name'),
            'ac_name' => $request->get('ac_name'),
            'ac_number' => $request->get('ac_number'),
            'district_name' => $request->get('district_name'),
            'reference' => $request->get('reference'),
            'amount' => $request->get('amount'),
            'commission' => $commissionAmount,
            'fee' => $chargeAmount,
        ]);

        $user->balance -= $finalAmount;
        $user->update();


        $notification = new FcmController();
        $notification->sendAdminNotification("Bank Transfer request", $request->get('bank_name') . ' transfer request from ' . $user->name);

        return response()->json([
            'status' => true,
            'message' => $request->get('bank_name') . ' transfer request send successful',
        ]);
    }

    public function billPayment(Request $request): JsonResponse
    {
        $amount = $request->get('amount');
        $finalAmount = $amount;

        $charge = Commission::findOrFail(9);
        $commissionRate = Commission::findOrFail(5);

        $chargeAmount = 0;
        $commissionAmount = 0;

        if ($charge->status === 'active') {
            $chargeAmount = $charge->amount;
            $finalAmount += $chargeAmount;
        }

        if ($commissionRate->status === 'active') {
            $commissionAmount = $amount * ($commissionRate->amount / 100);
            $finalAmount -= $commissionAmount;
        }

        $user = Auth::user();
        if ($user->pin != $request->pin) {
            return response()->json([
                'status' => false,
                'errors' => 'Your pin is wrong'
            ]);
        }

        if ($user->balance < $finalAmount) {
            return response()->json([
                'status' => false,
                'errors' => 'Insufficient balance'
            ]);
        }

        BillPayment::create([
            'user_id' => $user->id,
            'billing_name' => $request->get('billing_name'),
            'account_number' => $request->get('account_number'),
            'billing_date' => $request->get('billing_date'),
            'customer_name' => $request->get('customer_name'),
            'customer_number' => $request->get('customer_number'),
            'amount' => $request->get('amount'),
            'commission' => $commissionAmount,
            'fee' => $chargeAmount,
        ]);

        $user->balance -= $finalAmount;
        $user->update();

        $notification = new FcmController();
        $notification->sendAdminNotification("Bill Pay request", $request->get('billing_name') . ' request from ' . $user->name);

        return response()->json([
            'status' => true,
            'message' => $request->get('billing_name') . ' bill request send successful',
        ]);
    }

   public function rechargeRequest(Request $request): JsonResponse
{
    $user = auth()->user();
    $calculate = 0; // No commission calculation
    $finalAmount = $request->get('amount'); // Full amount to deduct from user balance

    if ($user->balance < $finalAmount) {
        return response()->json([
            'status' => false,
            'errors' => 'Insufficient balance'
        ]);
    }

    $number = $request->get('number');
    $type = $request->get('type');
    $operator = $request->get('operator_name');
    $amount = $request->get('amount');
    $sub_operator = $request->get('sub_operator');

    $client = new Client();
    $apiRecharge = APIRecharge::first();
    $apiUrl = $apiRecharge->api_url . '/api/recharge/create';
    $selected_type = ($type === 'Prepaid') ? 1 : (($type === 'Postpaid') ? 2 : null);

    if (!empty($apiRecharge->api_url) && !empty($apiRecharge->api_key)) {
        try {
            $response = $client->post($apiUrl, [
                'headers' => [
                    'Content-Type' => 'application/json',
                    'CD-API-KEY' => $apiRecharge->api_key,
                ],
                'json' => [
                    'number' => $number,
                    'type' => $selected_type,
                    'operator' => $sub_operator,
                    'amount' => $amount
                ]
            ]);

            $result = json_decode($response->getBody(), true);
            if (isset($result['status']) && $result['status'] == true) {
                $user->balance -= $finalAmount;
                $user->save();

                Recharge::create([
                    'user_id' => $user->id,
                    'number' => $number,
                    'operator_name' => $operator,
                    'sub_operator' => $sub_operator,
                    'type' => $type,
                    'amount' => $amount,
                    'commission' => $calculate,
                    'trx_id' => $result['trx_id'],
                    'status' => 'pending'
                ]);

                $notification = new FcmController();
                $notification->sendAdminNotification("Recharge request", $operator . ' recharge request from ' . $user->name);

                return response()->json([
                    'status' => true,
                    'message' => $result['message'],
                ]);
            } else {
                return response()->json([
                    'status' => false,
                    'errors' => $result['message'] ?? 'Failed to process recharge request'
                ]);
            }
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'errors' => $e->getMessage()
            ]);
        }
    } else {
        // Manual Recharge
        Recharge::create([
            'user_id' => $user->id,
            'number' => $number,
            'sub_operator' => $sub_operator,
            'operator_name' => $operator,
            'type' => $type,
            'amount' => $amount,
            'trx_id' => $this->generateUniqueTransactionId(10),
            'commission' => $calculate,
            'status' => 'pending'
        ]);

        $user->balance -= $finalAmount;
        $user->save();

        $notification = new FcmController();
        $notification->sendAdminNotification("Recharge request", $operator . ' recharge request from ' . $user->name);

        return response()->json([
            'status' => true,
            'message' => 'Flexi load request sent successfully',
        ]);
    }

    return response()->json([
        'status' => false,
        'errors' => 'Unexpected error occurred. Please try again.'
    ]);
}
}
