<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\Settings;
use App\Models\User;
use App\Models\UserDocument;
use App\Providers\RouteServiceProvider;
use Illuminate\Auth\Events\Registered;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Illuminate\Validation\Rules;
use Illuminate\View\View;

class RegisteredUserController extends Controller
{
    /**
     * Display the registration view.
     */
    public function create(): View
    {
        $settings = Settings::first();
        return view('auth.register', compact('settings'));
    }

    /**
     * Handle an incoming registration request.
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function store(Request $request): RedirectResponse
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'lowercase', 'email', 'max:255', 'unique:' . User::class],
            'phone' => ['required', 'string', 'max:255', 'unique:' . User::class],
            'document' => ['required'],
            'front_img' => 'required|image|mimes:jpg,jpeg,png',
            'back_img' => $request->document !== 'Birth Certificate Document' ? 'required|image|mimes:jpg,jpeg,png' : 'nullable',
            'pin' => ['required', 'integer', 'digits:4'],
            'password' => ['required', Rules\Password::defaults()],
        ]);

        $usernameBase = Str::lower(str_replace(' ', '', $request->name));
        $username = $usernameBase;
        $counter = 1;
        while (User::where('username', $username)->exists()) {
            $username = $usernameBase . $counter;
            $counter++;
        }

        $user = User::create([
            'name' => $request->name,
            'username' => $username,
            'email' => $request->email,
            'phone' => $request->phone,
            'pin' => $request->pin,
            'password' => Hash::make($request->password),
        ]);

        $documentUpload = new UserDocument();
        $documentUpload->user_id = $user->id;
        $documentUpload->type = $request->document;

        if ($request->hasFile('front_img')) {
            $documentUpload->document_one = $this->processImage($request->file('front_img'), '_front_', $request->phone);
        }
        if ($request->hasFile('back_img')) {
            $documentUpload->document_two = $this->processImage($request->file('back_img'), '_back_', $request->phone);
        }

        $documentUpload->save();

        event(new Registered($user));

        Auth::login($user);

        $notification = [
            'message' => 'আপনার রেজিস্ট্রেশন সফল হয়েছে.',
            'alert-type' => 'success'
        ];

        return redirect()->back()->with($notification);
    }

    private function processImage($file, $fieldName, $phone): string
    {
        $fileName = $phone .'_'. $fieldName . time().'.'. $file->getClientOriginalExtension();
        $path = 'upload/documents/' . $fileName;
        $file->move(public_path('upload/documents/'), $fileName);
        return $path;
    }

}
