<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\ResetPassword;
use App\Models\Settings;
use App\Models\SmsSetting;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Http;

class ResetPasswordController extends Controller
{
    public function showForgotForm()
    {
        $settings = Settings::findOrFail(1);
        return view('auth.forgot-password', compact('settings'));
    }

    public function sendOtp(Request $request): JsonResponse
    {
        $request->validate([
            'phone' => 'required|string',
        ]);

        // 5 মিনিটের পুরানো OTP ডিলিট
        $formatted = Carbon::now()->subMinutes(5)->toDateTimeString();
        ResetPassword::where('created_at', '<=', $formatted)->delete();

        // আগের OTP আছে কিনা চেক
        $existingOTP = ResetPassword::where('number', $request->phone)->first();
        if ($existingOTP) {
            return response()->json([
                'status' => false,
                'already_used' => true,
                'data' => [
                    'error' => 'An OTP has already been sent to this mobile number. Please check your messages or wait until it expires.',
                ]
            ]);
        }

        // ইউজার আছে কিনা চেক
        $user = User::where('phone', $request->phone)->first();
        if (!$user) {
            return response()->json([
                'status' => false,
                'data' => [
                    'error' => 'This mobile number was not found. Please verify and try again.',
                ]
            ]);
        }

        // SMS সার্ভিস সেটিংস চেক
        $service = SmsSetting::find(1);
        if (!$service || !$service->api_key || !$service->sender_id) {
            return response()->json([
                'status' => false,
                'data' => [
                    'error' => 'Configuration Problem, Contact to admin',
                ]
            ]);
        }

        // 4 ডিজিট OTP জেনারেট
        $otp = sprintf('%04d', rand(0, 9999));

        // OTP ডাটাবেজে ইনসার্ট
        ResetPassword::create([
            'number' => $request->phone,
            'otp' => $otp,
            'created_at' => Carbon::now()
        ]);

        // SMS মেসেজ
        $api_key = $service->api_key;
        $sender_id = $service->sender_id;
        $message = "Dear, " . $user->name . "\nYour OTP is $otp. Please use this code within 5 minutes to complete your verification.";

        // SMS API কল
        $response = Http::get("http://bulksmsbd.net/api/smsapi", [
            'api_key' => $api_key,
            'senderid' => $sender_id,
            'number' => $request->phone,
            'message' => $message,
            'type' => 'text'
        ]);

        if ($response->successful()) {
            return response()->json([
                'status' => true,
                'data' => [
                    'success' => 'An OTP has been sent to your mobile number. It will expire in 5 minutes.',
                ]
            ]);
        } else {
            // OTP ইনসার্ট করা হলেও SMS ফেইল করলে, অপশনally DB থেকে ডিলিট করতে পারেন
            ResetPassword::where('number', $request->phone)->delete();

            return response()->json([
                'status' => false,
                'data' => [
                    'error' => 'Failed to send OTP. Please try again later.',
                ]
            ]);
        }
    }

    public function verifyOtp(Request $request): JsonResponse
    {
        $request->validate([
            'phone' => 'required|string',
            'otp' => 'required|digits:4',
        ]);

        // 5 মিনিট আগের OTP ডিলিট
        $formatted = Carbon::now()->subMinutes(5)->toDateTimeString();
        ResetPassword::where('created_at', '<=', $formatted)->delete();

        // OTP যাচাই
        $otpValidation = ResetPassword::where('number', $request->phone)
            ->where('otp', $request->otp)
            ->first();

        if (!$otpValidation) {
            return response()->json([
                'status' => false,
                'data' => [
                    'error' => 'Invalid OTP Or Expired',
                ]
            ]);
        }

        // যাচাই সফল হলে OTP ডিলিট
        ResetPassword::where('number', $request->phone)->delete();

        return response()->json([
            'status' => true,
            'data' => [
                'success' => 'Your OTP is Valid',
            ]
        ]);
    }

    public function changePassword(Request $request): JsonResponse
    {
        $request->validate([
            'phone' => 'required|string',
            'password' => 'required|string|min:6|confirmed',
            'service_code' => 'required|integer',
        ]);

        if ($request->get('service_code') != 6484) {
            return response()->json([
                'status' => false,
                'errors' => 'Unauthorized access',
            ]);
        }

        $user = User::where('phone', $request->phone)->first();

        if (!$user) {
            return response()->json([
                'status' => false,
                'errors' => 'User not found',
            ]);
        }

        $user->password = Hash::make($request->get('password'));
        $user->save();

        return response()->json([
            'status' => true,
            'message' => 'Your new password has been updated',
        ]);
    }


}
