<?php

namespace App\Http\Controllers;

use App\Models\Settings;
use App\Models\User;
use Google\Client as GoogleClient;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\View\View;

class FcmController extends Controller
{
    public function index(): View
    {
        $dashboard = User::where('role', 'admin')->where('id', auth()->id())->first();
        $settings = Settings::findOrFail(1);

        return view('support.notification', compact('dashboard', 'settings'));
    }

    public function updateToken(Request $request): JsonResponse
    {
        $request->validate([
            'fcm_token' => 'required|string',
        ]);

        $user = User::where('role', 'admin')->findOrFail(Auth::id());

        if (!$user) {
            Log::error("User not found for ID: " . Auth::id());
            return response()->json(['message' => 'User not authenticated'], 401);
        }

        $user->fcm_token = $request->fcm_token;
        $user->save();

        return response()->json(['message' => 'Device token updated successfully']);
    }

    public function updateUserToken(Request $request): JsonResponse
    {
        $request->validate([
            'fcm_token' => 'required|string',
        ]);

        $user = User::where('role', 'retailer')->findOrFail(Auth::id());

        if (!$user) {
            Log::error("User not found for ID: " . Auth::id());
            return response()->json(['message' => 'User not authenticated'], 401);
        }

        $user->fcm_token = $request->fcm_token;
        $user->save();

        return response()->json(['message' => 'Device token updated successfully']);
    }

    public function updateFirebase(Request $request): RedirectResponse
    {
        // Validate the request data
        $request->validate([
            'firebase_api_key' => 'required|string',
            'firebase_auth_domain' => 'required|string',
            'firebase_project_id' => 'required|string',
            'firebase_messaging_sender_id' => 'required|string',
            'firebase_app_id' => 'required|string',
            'firebase_measurement_id' => 'required|string',
            'firebase_vapi_key' => 'required|string',
        ]);

        // Update the .env file
        $this->updateEnv([
            'FIREBASE_API_KEY' => $request->firebase_api_key,
            'FIREBASE_AUTH_DOMAIN' => $request->firebase_auth_domain,
            'FIREBASE_PROJECT_ID' => $request->firebase_project_id,
            'FIREBASE_MESSAGING_SENDER_ID' => $request->firebase_messaging_sender_id,
            'FIREBASE_APP_ID' => $request->firebase_app_id,
            'FIREBASE_MEASUREMENT_ID' => $request->firebase_measurement_id,
            'FIREBASE_WEB_PUSH_CERTIFICATE_KEY' => $request->firebase_vapi_key,
        ]);
        $notification = [
            'message' => 'Firebase configuration updated successfully.',
            'alert-type' => 'success'
        ];
        return redirect()->back()->with($notification);
    }

    protected function updateEnv(array $data): void
    {
        $path = base_path('.env');
        $content = File::get($path);

        foreach ($data as $key => $value) {
            $content = preg_replace("/^{$key}=.*$/m", "{$key}={$value}", $content);
        }

        File::put($path, $content);
        // Reload the configuration
        config()->set(array_merge(config()->all(), $data));
    }

    public function uploadJson(Request $request): RedirectResponse
    {
        $request->validate([
            'json_file' => 'required|file|mimes:json|max:2048', // 2MB max size
        ]);
        $path = $request->file('json_file')->storeAs('json', 'firebase_credentials.json');
        if ($path) {
            $notification = [
                'message' => 'JSON file uploaded successfully!',
                'alert-type' => 'success'
            ];
            return redirect()->back()->with($notification);
        } else {
            $notification = [
                'message' => 'Failed to upload JSON file.',
                'alert-type' => 'error'
            ];
            return redirect()->back()->with($notification);
        }
    }

    public function sendAllDeviceNotification(Request $request): RedirectResponse
    {
        $users = User::whereNotNull('fcm_token')->get();

        $projectId = config('firebase.project_id');
        $credentialsFilePath = Storage::path('/json/firebase_credentials.json');

        $client = new GoogleClient();
        $client->setAuthConfig($credentialsFilePath);
        $client->addScope('https://www.googleapis.com/auth/firebase.messaging');
        $client->refreshTokenWithAssertion();
        $token = $client->getAccessToken();

        $access_token = $token['access_token'];

        $headers = [
            "Authorization: Bearer $access_token",
            'Content-Type: application/json'
        ];

        foreach ($users as $user) {
            $fcm = $user->fcm_token;

            $notification = [
                "title" => $request->title,
                "body" => $request->body,
            ];

            $data = [
                "message" => [
                    "token" => $fcm,
                    "notification" => $notification,
                ]
            ];

            $payload = json_encode($data);

            // Send the notification via cURL
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, "https://fcm.googleapis.com/v1/projects/{$projectId}/messages:send");
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
            curl_setopt($ch, CURLOPT_VERBOSE, true); // Enable verbose output for debugging

            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);

            $response = curl_exec($ch);
            $err = curl_error($ch);
            curl_close($ch);
        }

        $notification = [
            'message' => 'Notifications have been sent to all users',
            'alert-type' => 'success'
        ];

        return back()->with($notification);

    }

    public function sendFcmNotification($id, $title, $body): JsonResponse
    {
        $user = User::findOrFail($id);
        $fcm = $user->fcm_token;

        if (!$fcm) {
            return response()->json(['message' => 'User does not have a device token'], 400);
        }


        $projectId = config('firebase.project_id');

        $credentialsFilePath = Storage::path('/json/firebase_credentials.json');
        $client = new GoogleClient();
        $client->setAuthConfig($credentialsFilePath);
        $client->addScope('https://www.googleapis.com/auth/firebase.messaging');
        $client->refreshTokenWithAssertion();
        $token = $client->getAccessToken();

        $access_token = $token['access_token'];

        $headers = [
            "Authorization: Bearer $access_token",
            'Content-Type: application/json'
        ];


        $notification = [
            "title" => $title,
            "body" => $body,
        ];

        $data = [
            "message" => [
                "token" => $fcm,
                "notification" => $notification,
            ]
        ];
        $payload = json_encode($data);
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, "https://fcm.googleapis.com/v1/projects/{$projectId}/messages:send");
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
        curl_setopt($ch, CURLOPT_VERBOSE, true); // Enable verbose output for debugging
        $response = curl_exec($ch);
        $err = curl_error($ch);
        curl_close($ch);

        if ($err) {
            return response()->json([
                'message' => 'Curl Error: ' . $err
            ], 500);
        } else {
            return response()->json([
                'message' => 'Notification has been sent',
                'response' => json_decode($response, true)
            ]);
        }
    }

    public function sendAdminNotification($title, $body): JsonResponse
    {
        // Get all admin users with a valid FCM token
        $admins = User::where('role', 'admin')->whereNotNull('fcm_token')->get();

        if ($admins->isEmpty()) {
            return response()->json(['message' => 'No admin users have a device token'], 400);
        }

        $projectId = config('firebase.project_id');
        $credentialsFilePath = Storage::path('/json/firebase_credentials.json');
        $client = new GoogleClient();
        $client->setAuthConfig($credentialsFilePath);
        $client->addScope('https://www.googleapis.com/auth/firebase.messaging');
        $client->refreshTokenWithAssertion();
        $token = $client->getAccessToken();
        $access_token = $token['access_token'];

        $headers = [
            "Authorization: Bearer $access_token",
            'Content-Type: application/json'
        ];

        // Prepare the notification data
        $notification = [
            "title" => $title,
            "body" => $body,
        ];

        // Iterate through each admin and send notifications
        foreach ($admins as $admin) {
            $fcm = $admin->fcm_token; // Get the FCM token for the current admin

            $data = [
                "message" => [
                    "token" => $fcm,
                    "notification" => $notification,
                ]
            ];
            $payload = json_encode($data);

            // Initialize cURL to send the notification
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, "https://fcm.googleapis.com/v1/projects/{$projectId}/messages:send");
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
            curl_setopt($ch, CURLOPT_VERBOSE, true); // Enable verbose output for debugging

            $response = curl_exec($ch);
            $err = curl_error($ch);
            curl_close($ch);
        }

        return response()->json([
            'message' => 'Notifications have been sent to all admins'
        ]);
    }


}
