<?php

namespace App\Http\Controllers;

use App\Models\Config;
use App\Models\Settings;
use App\Models\User;
use App\Services\EncryptionService;
use Illuminate\Contracts\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\HttpFoundation\BinaryFileResponse;

class InstallerController extends Controller
{
    public function requirements(): RedirectResponse|View
    {
        if (env('APP_REQUIREMENTS') == '1') {
            return redirect()->route('installer.database');
        }

        $requirements = [
            'PHP >= 8.2' => version_compare(PHP_VERSION, '8.2', '>='),
            'OpenSSL Extension' => extension_loaded('openssl'),
            'PDO Extension' => extension_loaded('pdo'),
            'Mbstring Extension' => extension_loaded('mbstring'),
            'Tokenizer Extension' => extension_loaded('tokenizer'),
            'JSON Extension' => extension_loaded('json'),
            'BCMath Extension' => extension_loaded('bcmath'),
            'DOM Extension' => extension_loaded('dom'),
            'PDO MySQL Extension' => extension_loaded('pdo_mysql'),
            'FileInfo Extension' => extension_loaded('fileinfo'),
        ];

        $isEnabled = !in_array(false, $requirements);

        return view('installer.requirements', compact('requirements', 'isEnabled'));
    }

    public function updateRequirements(): RedirectResponse
    {
        Artisan::call('key:generate', ['--no-interaction' => true]);
        $this->setSingleEnvironmentVariable('APP_REQUIREMENTS', '1');
        return Redirect::route('installer.database');
    }

    public function database(): View|RedirectResponse
    {
        if (env('APP_DATABASE') == '1') {
            return redirect()->route('installer.import');
        }

        return view('installer.database');
    }

    /**
     * @throws \Exception
     */
    public function install(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'db_host' => 'required',
            'db_name' => 'required',
            'db_user' => 'required',
            'db_password' => 'nullable',
        ], [
            'db_host.required' => 'Hostname is required.',
            'db_name.required' => 'Database name is required.',
            'db_user.required' => 'Database username is required.',
        ]);

        if ($validator->fails()) {
            return Redirect::back()->withErrors($validator)->withInput();
        }

        $this->setEnv([
            'DB_HOST' => $request->get('db_host'),
            'DB_DATABASE' => $request->get('db_name'),
            'DB_USERNAME' => $request->get('db_user'),
            'DB_PASSWORD' => $request->get('db_password'),
        ]);

        try {
            $this->setSingleEnvironmentVariable('APP_DATABASE', '1');
            return redirect()->back()->with('success', 'Database connected successfully.');
        } catch (\Exception $e) {
            Log::error('Database Connection Error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Could not connect to the database. Please check your credentials.')->withInput();
        }
    }

    public function import(): View|RedirectResponse
    {
        if (env('APP_IMPORT') == '1') {
            return redirect()->route('installer.finished');
        }

        return view('installer.import');
    }

    public function dbDownload(): BinaryFileResponse|RedirectResponse
    {
        $path = storage_path('app/db.sql');

        if (file_exists($path)) {
            return response()->download($path, 'database.sql');
        } else {
            return redirect()->back()->with('error', 'SQL file not found.');
        }
    }

    public function dbImport(): RedirectResponse
    {
        try {
            $path = storage_path('app/db.sql');

            if (!File::exists($path)) {
                return redirect()->back()->with('error', 'Database file not found.');
            }

            $sql = File::get($path);
            DB::unprepared($sql);
            $this->setSingleEnvironmentVariable('APP_IMPORT', '1');

            return redirect()->route('installer.finished');
        } catch (\Exception $e) {
            Log::error('Database Import Error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Could not import the database. Try to manual import');
        }
    }

    public function skipToFinish(): RedirectResponse
    {
        $this->setSingleEnvironmentVariable('APP_IMPORT', '1');
        return redirect()->route('installer.finished');
    }

    public function finished(): View|RedirectResponse
    {
        if (env('APP_FINISHED') == '1') {
            return redirect()->route('login');
        }
        return view('installer.finish');
    }

    /**
     * @throws \Exception
     */
    public function installationFinished(Request $request): RedirectResponse
    {
        // Step 1: Check if installation is already finished
        if (env('APP_INSTALLED') == '1' && env('APP_FINISHED') == '1') {
            return redirect('/')->with('info', 'The application is already installed.');
        }

        $validator = Validator::make($request->all(), [
            'app_name' => 'required',
            'app_package_name' => 'required',
            'app_url' => 'required',
            'admin_email' => 'required|email',
            'admin_password' => 'required|min:8',
            'admin_confirm_password' => 'required|min:8|same:admin_password',
        ], [
            'app_name.required' => 'Admin Panel name is required.',
            'app_package_name.required' => 'App package name is required.',
            'app_url.required' => 'Admin URL is required.',
            'admin_email.required' => 'Admin email is required.',
            'admin_email.email' => 'Admin email must be a valid email address.',
            'admin_password.required' => 'Admin password is required.',
            'admin_password.min' => 'Admin password must be at least 8 characters.',
            'admin_confirm_password.required' => 'Admin confirm password is required.',
            'admin_confirm_password.min' => 'Admin confirm password must be at least 8 characters.',
            'admin_confirm_password.same' => 'Admin confirm password does not match.',
        ]);

        if ($validator->fails()) {
            return Redirect::back()->withErrors($validator)->withInput();
        }

        $secretKey = EncryptionService::generateKey($request->get('app_package_name'), 256);
        $appSecret = EncryptionService::encrypt($request->get('app_url'), $secretKey);

        User::create([
            'email' => $request->get('admin_email'),
            'role' => 'admin',
            'password' => Hash::make($request->get('admin_password')),
        ]);


        $appName = str_replace(' ', '_', $request->get('app_name'));

        $settings = Settings::findOrFail(1);
        $settings->brandName = $request->get('app_name');
        $settings->update();

        $config = Config::findOrFail(1);
        $config->packageName = $request->get('app_package_name');
        $config->app_secret = $appSecret;
        $config->secret_key = $secretKey;
        $config->update();

        if ($config) {
            $this->setEnv([
                'APP_NAME' => $appName,
                'APP_URL' => $request->get('app_url'),
                'APP_PACKAGE_NAME' => $request->get('app_package_name'),
                'APP_FINISHED' => '1',
                'APP_INSTALLED' => '1',
            ]);

            $this->installerAndRoutes();

            return redirect('/')->with('success', 'Installation successful. Please login.');
        } else {
            return redirect('/')->with('error', 'Installation failed. Please try again.');
        }

    }

    protected function installerAndRoutes(): void
    {
        $files = [
            app_path('Http/Controllers/InstallerController.php'),
            app_path('Services/EncryptionService.php'),
            resource_path('views/installer'),
            base_path('routes/installer.php'),
            storage_path('app/db.sql'),
        ];

        foreach ($files as $file) {
            if (File::exists($file)) {
                if (is_dir($file)) {
                    File::deleteDirectory($file);
                } else {
                    File::delete($file);
                }
            }
        }
        Artisan::call('config:clear');
        Artisan::call('route:clear');
    }

    protected function setSingleEnvironmentVariable($key, $value): void
    {
        $path = base_path('.env');

        if (File::exists($path)) {
            $env = File::get($path);
            $pattern = "/^$key=.*/m";
            $replacement = "$key=$value";

            if (preg_match($pattern, $env)) {
                $env = preg_replace($pattern, $replacement, $env);
            } else {
                $env .= "\n$replacement";
            }

            File::put($path, $env);

            Artisan::call('config:clear');
            Artisan::call('cache:clear');
        }
    }

    protected function setEnv(array $data): void
    {
        $envPath = base_path('.env');

        if (!file_exists($envPath)) {
            throw new \Exception('.env file does not exist');
        }

        $env = file_get_contents($envPath);
        foreach ($data as $key => $value) {
            if (preg_match('/^' . $key . '=.*/m', $env)) {
                $env = preg_replace('/^' . $key . '=.*/m', $key . '=' . $value, $env);
            } else {
                $env .= "\n" . $key . '=' . $value;
            }
        }
        file_put_contents($envPath, $env);
        Artisan::call('config:clear');
    }

}
