<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\Commission;
use App\Models\DepositCharge;
use App\Models\GatewaySetup;
use App\Models\MinimumDeposit;
use App\Models\Payment;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class PaymentController extends Controller
{
    private $apiKey;
    private $successUrl;
    private $cancelUrl;
    private $webhookUrl;

    public function __construct()
    {
        $gateway = GatewaySetup::first();
        $this->apiKey = $gateway->brand_key;
        $this->successUrl = route('user.payment.success');
        $this->cancelUrl = route('user.payment.cancel');
        $this->webhookUrl = route('user.payment.webhook');
    }

    public function create(Request $request): RedirectResponse
    {
        $amount = $request->input('amount');
        $payableAmount = $amount;

        $minimumDeposit = MinimumDeposit::first();
        $depositCharge = DepositCharge::first();

        if ($minimumDeposit && $minimumDeposit->status === 'active') {
            if ($amount < $minimumDeposit->minimum_deposit) {
                $notification = [
                    'message' => 'Minimum deposit is ' . $minimumDeposit->minimum_deposit . ' BDT',
                    'alert-type' => 'error'
                ];
                return back()->with($notification);
            }
        }

        if ($depositCharge && $depositCharge->status === 'active') {
            $chargePercent = $depositCharge->deposit_charge;
            $chargeAmount = ($amount * $chargePercent) / 100;
            $payableAmount = $amount + $chargeAmount;
        }
        $response = Http::timeout(60)->withHeaders([
            'API-KEY' => $this->apiKey,
            'Content-Type' => 'application/json',
        ])->post('https://pay.bdautopay.com/api/payment/create', [
            'success_url' => $this->successUrl,
            'cancel_url' => $this->cancelUrl,
            'webhook_url' => $this->webhookUrl,
            'cus_name' => Auth::user()->name,
            'cus_email' => Auth::user()->email,
            'metadata' => [
                'user_id' => Auth::id(),
                'phone' => Auth::user()->phone,
                'original_amount' => $amount,
                'charge_amount' => $payableAmount - $amount,
            ],
            'amount' => $payableAmount,
        ]);

        $responseData = json_decode($response->body(), true);

        if (isset($responseData['status']) && $responseData['status'] == 1) {
            return redirect()->away($responseData['payment_url']);
        } else {
            $notification = [
                'message' => '❌ Payment initiation failed!',
                'alert-type' => 'error'
            ];
            return back()->with($notification);
        }
    }

    public function cancel(Request $request): RedirectResponse
    {
        return redirect()->route('user.dashboard')->with('error', '❌ Payment cancelled!');
    }

   public function success(Request $request): RedirectResponse
    {
        $transactionId = trim($request->input('transactionId'));
        $status = $request->input('status');
        $method = $request->input('paymentMethod');

        $paymentVerification = $this->verifyPayment($transactionId);
        Log::info('🛡️ Payment verification response:', $paymentVerification);

        if ($paymentVerification['status'] !== 'COMPLETED') {
            return redirect()->route('user.dashboard')->with([
                'message' => '❌ Payment verification failed!',
                'alert-type' => 'error'
            ]);
        }

        $metadataRaw = $paymentVerification['metadata'] ?? null;
        $metadata = [];

        if (is_string($metadataRaw)) {
            $metadata = json_decode($metadataRaw, true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                $metadata = [];
            }
        } elseif (is_array($metadataRaw)) {
            $metadata = $metadataRaw;
        }

        if (!isset($metadata['original_amount'])) {
            return redirect()->route('user.dashboard')->with([
                'message' => '❌ Payment details incomplete!',
                'alert-type' => 'error'
            ]);
        }

        $originalAmount = $metadata['original_amount'];

        $payment = Payment::firstOrCreate(
            ['transaction_id' => $transactionId],
            [
                'user_id' => auth()->id(),
                'amount' => $originalAmount,
                'method' => $method,
                'status' => 'completed'
            ]
        );

        if ($payment->wasRecentlyCreated) {
            $user = User::findOrFail(auth()->id());

            $commissionRate = Commission::first();
            if ($commissionRate && $commissionRate->status === 'On') {
                $calculate = $originalAmount * ($commissionRate->commission / 100);
                $finalAmount = $originalAmount + $calculate;
            } else {
                $finalAmount = $originalAmount;
            }

            $user->increment('balance', $finalAmount);
        }

        return redirect()->route('user.dashboard')->with([
            'message' => '✅ Payment successful!',
            'alert-type' => 'success'
        ]);
    }

    public function webhook(Request $request): JsonResponse
    {
        Log::info('📩 Webhook received:', $request->all());

        return response()->json(['status' => 'received'], 200);
    }
    private function verifyPayment($transactionId)
    {
        try {
            $response = Http::timeout(120)->withOptions([
                'verify' => false,
            ])->withHeaders([
                'API-KEY' => $this->apiKey,
                'Content-Type' => 'application/json',
            ])->post('https://pay.bdautopay.com/api/payment/verify', [
                'transaction_id' => $transactionId,
            ]);

            $result = json_decode($response->body(), true);
            return $result;
        } catch (\Exception $e) {
            return ['status' => 'failed', 'message' => $e->getMessage()];
        }
    }
}
