<?php

namespace App\Http\Middleware;

use Illuminate\Auth\Middleware\Authenticate as Middleware;
use Illuminate\Http\Request;
use Closure;
use Illuminate\Support\Facades\Auth;
use Laravel\Sanctum\PersonalAccessToken;

class Authenticate extends Middleware
{
    /**
     * Handle an incoming request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Closure  $next
     * @param  mixed  ...$guards
     * @return mixed
     */
    public function handle($request, Closure $next, ...$guards)
    {
        if (empty($guards)) {
            $guards = [null];
        }

        // Check if the user is authenticated
        foreach ($guards as $guard) {
            if (Auth::guard($guard)->check()) {
                $this->auth->shouldUse($guard);
                return $next($request);
            }
        }

        // Handle API requests
        if ($request->is('api/*')) {
            // Retrieve the Bearer token
            $token = $request->bearerToken();

            if (!$token) {
                return response()->json([
                    'status' => false,
                    'message' => 'Unauthorized',
                    'errors' => [
                        'session' => 'Unauthorized login detected'
                    ]
                ], 401);
            }

            // Validate the token
            $personalAccessToken = PersonalAccessToken::findToken($token);
            if (!$personalAccessToken) {
                return response()->json([
                    'status' => false,
                    'message' => 'Unauthorized',
                    'errors' => [
                        'session' => 'Unauthorized login detected'
                    ]
                ], 401);
            }

            // Check if the token has expired
            $expirationMinutes = config('sanctum.expiration');
            if ($expirationMinutes && $personalAccessToken->created_at->addMinutes($expirationMinutes)->isPast()) {
                return response()->json([
                    'status' => false,
                    'message' => 'Session Expired',
                    'errors' => [
                        'session' => 'Your session has expired. Please log in again.'
                    ]
                ], 401);
            }

            // If all checks pass, proceed with the request
            return $next($request);
        }

        // For web requests or other routes, handle as usual
        return $this->unauthenticated($request, $guards);
    }

    /**
     * Get the path the user should be redirected to when they are not authenticated.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return string|null
     */
    protected function redirectTo(Request $request): ?string
    {
        return $request->expectsJson() ? null : route('login');
    }
}
